#pragma once

#include <src/logic/db/contacts/create_contacts_list.hpp>
#include <src/logic/db/contacts/get_subscribed_list_ids_by_owner.hpp>
#include <src/logic/db/contacts/share_contacts_list.hpp>
#include <src/logic/db/contacts/subscribe_to_contacts_list.hpp>
#include <src/services/db/begin.hpp>
#include <src/services/db/commit.hpp>

namespace collie::logic::db::contacts {

template<typename OrgConnection, typename UserProvider> expected<Revision> subscribeToContacts(
        OrgConnection&& orgConnection,
        UserProvider&& userProvider,
        ListId orgListId,
        std::string clientListName) {
    auto transact{[&](auto&& transaction) {
        const auto ownerUserId{services::db::unwrap(orgConnection).uid()};
        std::string ownerUserType{"connect_organization"};
        const auto getSubscribedListIds{[&]{return getSubscribedListIdsByOwner(transaction, ownerUserId,
                ownerUserType);}};

        auto createUserListAndSubscribe{[&](const auto& subscribedListIds) {
            if (!subscribedListIds.empty()) {
                return make_expected();
            }

            const auto createUserList{[&]{return createContactsList(transaction, std::move(
                    clientListName));}};
            auto subscribe{[&](auto&& createdList){return subscribeToContactsList(transaction,
                    createdList.listId, ownerUserId, std::move(ownerUserType), orgListId);}};
            auto commit{[&](auto){return services::db::commit(userProvider, std::move(transaction)).bind(
                    [](auto&&){});}};

            return createUserList().bind(std::move(subscribe)).bind(std::move(commit));
        }};

        return getSubscribedListIds().bind(std::move(createUserListAndSubscribe));
    }};

    auto share{[&] {
        std::string clientUserType{"passport_user"};
        return shareContactsList(orgConnection, orgListId, services::db::unwrap(userProvider).uid(),
                std::move(clientUserType));
    }};

    return services::db::begin(userProvider).bind(std::move(transact)).bind(std::move(share));
}

}
