#pragma once

#include <sharpei_client/errors.h>
#include <src/logic/db/contacts/delete_contacts_list.hpp>
#include <src/logic/db/contacts/get_subscribed_list_ids_by_owner.hpp>
#include <src/logic/db/contacts/revoke_contacts_list.hpp>
#include <src/logic/db/contacts/revoke_subscribed_contacts_list.hpp>
#include <src/services/db/begin.hpp>
#include <src/services/db/commit.hpp>

namespace collie::logic::db::contacts {

template<typename OrgConnection, typename UserProvider> expected<Revision> unsubscribeFromContacts(
        OrgConnection&& orgConnection, UserProvider&& userProvider, ListId orgListId) {
    auto transact{[&](auto&& transaction) {
        const auto ownerUserId{services::db::unwrap(orgConnection).uid()};
        std::string ownerUserType{"connect_organization"};
        const auto getSubscribedListIds{[&]{return getSubscribedListIdsByOwner(transaction, ownerUserId,
                ownerUserType);}};

        auto unsubscribeAndDeleteUserList{[&](const auto& subscribedListIds) {
            if (subscribedListIds.empty()) {
                return make_expected();
            }

            const auto unsubscribe{[&]{return revokeSubscribedContactsList(transaction,
                    subscribedListIds[0], ownerUserId, std::move(ownerUserType), orgListId);}};
            auto deleteUserList{[&](auto){return deleteContactsList(transaction, subscribedListIds[0]);}};
            auto commit{[&](auto){return services::db::commit(userProvider, std::move(transaction)).bind(
                    [](auto&&){});}};

            return unsubscribe().bind(std::move(deleteUserList)).bind(std::move(commit));
        }};

        return getSubscribedListIds().bind(std::move(unsubscribeAndDeleteUserList));
    }};

    auto unshare{[&] {
        std::string clientUserType{"passport_user"};
        return revokeContactsList(orgConnection, orgListId, services::db::unwrap(userProvider).uid(),
                std::move(clientUserType));
    }};

    auto expectedTransaction{services::db::begin(userProvider)};
    if (expectedTransaction) {
        return expectedTransaction.bind(std::move(transact)).bind(std::move(unshare));
    }

    if (expectedTransaction.error() == ::sharpei::client::Errors::UidNotFound) {
        return unshare();
    }

    return make_unexpected(expectedTransaction.error());
}

}
