#include "update_contacts.hpp"

#include <src/logic/interface/types/reflection/vcard.hpp>
#include <yamail/data/serialization/yajl.h>

#include <boost/range/adaptor/filtered.hpp>
#include <boost/range/algorithm/transform.hpp>

#include <unordered_set>

namespace collie::logic::db::contacts {

std::vector<std::int64_t> makeContactsIds(const std::vector<UpdatedContact>& values) {
    std::vector<std::int64_t> result;
    result.reserve(values.size());
    boost::transform(values, std::back_inserter(result), [&] (const auto& v) { return v.contact_id; });
    return result;
}

DbUpdatedContact makeUpdatedContact(const UpdatedContact& value) {
    using yamail::data::serialization::toJson;
    DbUpdatedContact result;
    result.contact_id = value.contact_id;
    result.list_id = value.list_id;
    if (value.vcard) {
        result.format = "vcard_v1";
        auto vcard = value.vcard;
        vcard->emails.reset();
        result.vcard = ozo::pg::jsonb(toJson(*vcard));
        result.uri = value.uri;
    }
    return result;
}

std::vector<DbUpdatedContact> makeUpdatedContacts(const std::vector<UpdatedContact>& values) {
    std::vector<DbUpdatedContact> result;
    result.reserve(values.size());
    boost::transform(values, std::back_inserter(result), makeUpdatedContact);
    return result;
}

EmailIdsMap makeEmailIdsMap(const std::vector<ContactIdEmailIdEmailRow>& rows) {
    EmailIdsMap result;
    boost::transform(rows, std::inserter(result, result.end()),
        [] (const auto& v) {
            return std::make_pair(std::make_pair(v.contact_id, std::string_view(v.email)), v.email_id);
        });
    return result;
}

namespace {

template <class Out>
void makeNewContactsEmails(const UpdatedContact& value, const EmailIdsMap& emailIdMap, Out out) {
    if (value.vcard && value.vcard->emails) {
        for (auto& email : *value.vcard->emails) {
            if (!email.email) {
                continue;
            }
            if (email.email.value().empty()) {
                continue;
            }
            if (emailIdMap.count({value.contact_id, std::string_view(*email.email)})) {
                continue;
            }
            NewContactsEmail newContactsEmail;
            newContactsEmail.contact_id = value.contact_id;
            newContactsEmail.email = email.email;
            newContactsEmail.type = email.type;
            newContactsEmail.label = email.label;
            *out++ = std::move(newContactsEmail);
        }
    }
}

} // namespace

std::vector<NewContactsEmail> makeNewContactsEmails(const std::vector<UpdatedContact>& updatedContacts,
        const EmailIdsMap& emailIdsMap) {
    std::vector<NewContactsEmail> result;
    for (const auto& v : updatedContacts) {
        makeNewContactsEmails(v, emailIdsMap, std::back_inserter(result));
    }
    return result;
}

namespace {

template <class Out>
void makeUpdatedContactsEmails(const UpdatedContact& value, const EmailIdsMap& emailIdMap, Out out) {
    if (value.vcard && value.vcard->emails) {
        for (auto& email : *value.vcard->emails) {
            if (!email.email) {
                continue;
            }
            const auto it = emailIdMap.find({value.contact_id, std::string_view(*email.email)});
            if (it == emailIdMap.end()) {
                continue;
            }
            UpdatedContactsEmail updateContactsEmail;
            updateContactsEmail.email_id = it->second;
            updateContactsEmail.email = email.email;
            updateContactsEmail.type = email.type;
            updateContactsEmail.label = email.label;
            *out++ = std::move(updateContactsEmail);
        }
    }
}

} // namespace

std::vector<UpdatedContactsEmail> makeUpdatedContactsEmails(const std::vector<UpdatedContact>& updatedContacts,
        const EmailIdsMap& emailIdsMap) {
    std::vector<UpdatedContactsEmail> result;
    for (const auto& v : updatedContacts) {
        makeUpdatedContactsEmails(v, emailIdsMap, std::back_inserter(result));
    }
    return result;
}

namespace {

template <class Out>
void makeRemovedContactsEmailIds(const UpdatedContact& value, const EmailIdsMap& emailIdMap, Out out) {
    if (value.vcard && value.vcard->emails) {
        std::unordered_set<std::string_view> vcardEmails;
        for (const auto& v : *value.vcard->emails) {
            if (v.email) {
                vcardEmails.insert(*v.email);
            }
        }

        auto filter{[&](const auto& element){return element.first.first == value.contact_id;}};
        for (const auto& v : emailIdMap | boost::adaptors::filtered(std::move(filter))) {
            if (!vcardEmails.count(v.first.second)) {
                *out++ = v.second;
            }
        }
    }
}

} // namespace

std::vector<std::int64_t> makeRemovedContactsEmailIds(const std::vector<UpdatedContact>& updatedContacts,
        const EmailIdsMap& emailIdsMap) {
    std::vector<std::int64_t> result;
    for (const auto& v : updatedContacts) {
        makeRemovedContactsEmailIds(v, emailIdsMap, std::back_inserter(result));
    }
    return result;
}

TaggedContacts makeTaggedContactIds(const std::vector<UpdatedContact>& updatedContacts) {
    TaggedContacts result;
    for (const auto& v : updatedContacts) {
        if (v.add_tag_ids) {
            for (const auto tagId : *v.add_tag_ids) {
                result[tagId].push_back(v.contact_id);
            }
        }
    }
    return result;
}

TaggedContacts makeUntaggedContactIds(const std::vector<UpdatedContact>& updatedContacts) {
    TaggedContacts result;
    for (const auto& v : updatedContacts) {
        if (v.remove_tag_ids) {
            for (const auto tagId : *v.remove_tag_ids) {
                result[tagId].push_back(v.contact_id);
            }
        }
    }
    return result;
}

} // namespace collie::logic::db::contacts
