#pragma once

#include "create_contacts_emails.hpp"
#include "get_email_ids_tag_ids_by_email_ids.hpp"
#include "remove_contacts_emails.hpp"
#include "tag_contacts_and_contacts_emails.hpp"
#include "untag_contacts_and_contacts_emails.hpp"
#include "update_contacts_emails.hpp"

#include <src/logic/interface/types/revision.hpp>
#include <src/logic/interface/types/updated_contacts.hpp>

#include <src/services/db/contacts/query.hpp>
#include <src/services/db/request.hpp>
#include <src/services/db/utils.hpp>

#include <src/expected.hpp>

#include <map>
#include <vector>

namespace collie::logic::db::contacts {

using DbUpdatedContact = services::db::contacts::UpdatedContact;
using services::db::contacts::ContactIdEmailIdEmailRow;
using services::db::contacts::NewContactsEmail;
using services::db::contacts::UpdatedContactsEmail;
using EmailIdsMap = std::map<std::pair<std::int64_t, std::string_view>, std::int64_t>;
using TaggedContacts = std::map<std::int64_t, std::vector<std::int64_t>>;

std::vector<std::int64_t> makeContactsIds(const std::vector<UpdatedContact>& values);

DbUpdatedContact makeUpdatedContact(const UpdatedContact& value);

std::vector<DbUpdatedContact> makeUpdatedContacts(const std::vector<UpdatedContact>& values);

EmailIdsMap makeEmailIdsMap(const std::vector<ContactIdEmailIdEmailRow>& rows);

std::vector<NewContactsEmail> makeNewContactsEmails(const std::vector<UpdatedContact>& updatedContacts,
        const EmailIdsMap& emailIdsMap);

std::vector<UpdatedContactsEmail> makeUpdatedContactsEmails(const std::vector<UpdatedContact>& updatedContacts,
        const EmailIdsMap& emailIdsMap);

std::vector<std::int64_t> makeRemovedContactsEmailIds(const std::vector<UpdatedContact>& updatedContacts,
        const EmailIdsMap& emailIdsMap);

TaggedContacts makeTaggedContactIds(const std::vector<UpdatedContact>& updatedContacts);

TaggedContacts makeUntaggedContactIds(const std::vector<UpdatedContact>& updatedContacts);

template<typename Connection> expected<Revision> updateContactsImpl(Connection&& connection,
        UpdatedContacts updatedContacts, std::optional<size_t> chunkSize) {
    using services::db::request;

    const auto context = services::db::unwrap(connection).context();
    const auto uid = services::db::unwrap(connection).uid();
    const auto userType = services::db::unwrap(connection).userType();
    const auto contactsIds = makeContactsIds(updatedContacts.updated_contacts);
    const auto contacts = makeUpdatedContacts(updatedContacts.updated_contacts);

    services::db::contacts::query::GetContactsEmailIdsByContactsIds query;
    query.uid = uid;
    query.user_type = userType;
    query.contacts_ids = contactsIds;

    const auto emailIds = request(connection, std::as_const(query));
    if (!emailIds) {
        return make_expected_from_error<Revision>(std::move(emailIds.error()));
    }

    const auto getSingleRow{[](auto&& rows){return expected(services::db::expectSingleRow(
            std::move(rows)));}};

    const auto updateContacts{[&] {
        services::db::contacts::query::UpdateContacts query;
        query.uid = uid;
        query.user_type = userType;
        query.contacts = contacts;
        query.x_request_id = context->requestId();
        query.revision = updatedContacts.revision;
        return request(connection, std::as_const(query)).bind(getSingleRow);
    }};

    const auto emailIdsMap = makeEmailIdsMap(emailIds.value());

    auto updateContactsEmails{[&](auto&& revision) {
        auto updatedEmails{makeUpdatedContactsEmails(updatedContacts.updated_contacts, emailIdsMap)};
        if (updatedEmails.empty()) {
            return make_expected(revision);
        }

        return contacts::updateContactsEmails(connection, std::move(updatedEmails), std::move(updatedContacts.
                revision), chunkSize);
    }};

    auto removeContactsEmails{[&](auto&& revision) {
        auto removedEmailIds{makeRemovedContactsEmailIds(updatedContacts.updated_contacts, emailIdsMap)};
        if (removedEmailIds.empty()) {
            return make_expected(revision);
        }

        auto makeEmailIdsByTagIdMap{[](const auto& emailIdsTagIds) {
            std::unordered_map<TagId, std::vector<ContactsEmailId>> emailIdsByTagId;
            for (const auto& element : emailIdsTagIds) {
                emailIdsByTagId[element.tag_id].emplace_back(ContactsEmailId{{}, element.email_id});
            }

            return emailIdsByTagId;
        }};

        auto untagContactsAndContactsEmails{[&](auto&& emailIdsByTagId) {
            auto result{make_expected(revision)};
            for (auto& element : emailIdsByTagId) {
                result = contacts::untagContactsAndContactsEmails(connection, element.first, {}, std::move(
                        element.second));
                if (!result) {
                    return result;
                }
            }

            return result;
        }};

        auto removeEmails{[&](auto){return contacts::removeContactsEmails(connection, std::move(
                removedEmailIds));}};
        return contacts::getEmailIdsTagIdsByEmailIds(connection, removedEmailIds).
                bind(std::move(makeEmailIdsByTagIdMap)).
                bind(std::move(untagContactsAndContactsEmails)).
                bind(std::move(removeEmails));
    }};

    auto createContactsEmails{[&](auto&& revision) {
        auto newEmails{makeNewContactsEmails(updatedContacts.updated_contacts, emailIdsMap)};
        if (newEmails.empty()) {
            return make_expected(revision);
        }

        auto getRevision{[](const auto& createdEmails){return createdEmails.revision;}};
        return contacts::createContactsEmails(connection, std::move(newEmails), chunkSize).bind(
                std::move(getRevision));
    }};

    auto tagContacts{[&](auto&& revision) {
        auto taggedContacts{makeTaggedContactIds(updatedContacts.updated_contacts)};
        if (taggedContacts.empty()) {
            return make_expected(revision);
        }

        auto result{make_expected(revision)};
        for (auto& element : taggedContacts) {
            result = contacts::tagContactsAndContactsEmails(connection, element.first, std::move(element.
                    second), {});
            if (!result) {
                return result;
            }
        }

        return result;
    }};

    auto untagContacts{[&](auto&& revision) {
        auto untaggedContacts{makeUntaggedContactIds(updatedContacts.updated_contacts)};
        if (untaggedContacts.empty()) {
            return make_expected(revision);
        }

        auto result{make_expected(revision)};
        for (auto& element : untaggedContacts) {
            result = contacts::untagContactsAndContactsEmails(connection, element.first, std::move(element.
                    second), {});
            if (!result) {
                return result;
            }
        }

        return result;
    }};

    return updateContacts().
            bind(std::move(updateContactsEmails)).
            bind(std::move(removeContactsEmails)).
            bind(std::move(createContactsEmails)).
            bind(std::move(tagContacts)).
            bind(std::move(untagContacts));
}

template<typename Connection> expected<Revision> updateContacts(UpdatedContacts updatedContacts,
        Connection&& connection) {
    return updateContactsImpl(connection, std::move(updatedContacts), {});
}

template<typename Connection> expected<Revision> updateContacts(UpdatedContacts updatedContacts,
        Connection&& connection, size_t chunkSize) {
    auto result{updatedContacts.revision ? expected<Revision>(*updatedContacts.revision) :
            expected<Revision>()};
    UpdatedContacts chunk;
    chunk.revision = std::move(updatedContacts.revision);
    const auto& contacts = updatedContacts.updated_contacts;
    for (std::size_t index{0}; index < contacts.size(); ++index) {
        chunk.updated_contacts.emplace_back(std::move(contacts[index]));
        if ((chunk.updated_contacts.size() == chunkSize) || (index == (contacts.size() - 1))) {
            result = updateContactsImpl(connection, chunk, chunkSize);
            if (!result) {
                return result;
            }

            if (chunk.revision && (result.value() > *chunk.revision)) {
                chunk.revision = result.value();
            }

            chunk.updated_contacts.clear();
        }
    }

    return result;
}

} // namespace collie::logic::db::contacts
