#pragma once

#include <src/expected.hpp>
#include <src/logic/interface/types/revision.hpp>
#include <src/services/db/contacts/query.hpp>
#include <src/services/db/request.hpp>
#include <src/services/db/utils.hpp>

namespace collie::logic::db::contacts {

template<typename Connection> expected<Revision> updateContactsEmailsImpl(Connection&& connection,
        std::vector<services::db::contacts::UpdatedContactsEmail> emails, std::optional<Revision> revision) {
    services::db::contacts::query::UpdateContactsEmails query;
    const auto uid{services::db::unwrap(connection).uid()};
    query.uid = uid;
    query.user_type = services::db::unwrap(connection).userType();
    query.emails = std::move(emails);
    const auto context{services::db::unwrap(connection).context()};
    query.x_request_id = context->requestId();
    query.revision = std::move(revision);
    return services::db::request(connection, std::as_const(query)).bind([&](auto&& rows) {
        const auto result{services::db::expectSingleRow(std::move(rows))};
        LOGDOG_(context->logger(), notice, log::query=ozo::get_query_name(query),
                log::uid=std::to_string(uid), log::revision=result);
        return result;
    });
}

template<typename Connection> expected<Revision> updateContactsEmailsImpl(
        Connection&& connection,
        std::vector<services::db::contacts::UpdatedContactsEmail> emails,
        std::optional<Revision> revision,
        size_t chunkSize) {
    auto begin{emails.begin()};
    auto result{revision ? expected<Revision>{*revision} : expected<Revision>{}};
    while (begin != emails.end()) {
        auto end{(static_cast<size_t>(std::distance(begin, emails.end())) >= chunkSize) ? begin + chunkSize :
                emails.end()};
        result = updateContactsEmailsImpl(connection, {std::make_move_iterator(begin),
                std::make_move_iterator(end)}, revision);
        if (!result) {
            return result;
        }

        begin = end;
        if (revision && (*result > *revision)) {
            revision = *result;
        }
    }

    return result;
}

template<typename Connection> expected<Revision> updateContactsEmails(
        Connection&& connection,
        std::vector<services::db::contacts::UpdatedContactsEmail> emails,
        std::optional<Revision> revision,
        const std::optional<size_t>& chunkSize) {
    if (chunkSize) {
        return updateContactsEmailsImpl(connection, std::move(emails), std::move(revision), *chunkSize);
    } else {
        return updateContactsEmailsImpl(connection, std::move(emails), std::move(revision));
    }
}

}
