#pragma once

#include "contacts/create_contacts.hpp"

#include <src/logic/interface/create_contacts.hpp>
#include <src/services/db/begin.hpp>
#include <src/services/db/commit.hpp>
#include <src/services/db/connection_provider.hpp>
#include <src/services/db/contacts/query.hpp>
#include <src/services/db/passport_user_id.hpp>
#include <src/services/db/utils.hpp>

namespace collie::logic::db {

template <class MakeConnectionProvider>
class CreateContactsImpl final : public CreateContacts {
public:
    explicit CreateContactsImpl(MakeConnectionProvider makeConnectionProvider)
        : makeConnectionProvider(std::move(makeConnectionProvider)) {
    }

    expected<CreatedContacts> operator()(const TaskContextPtr& context, const Uid& uid,
            std::vector<NewContact> newContacts) const override {
        using services::db::begin;
        using services::db::commit;

        std::int64_t convertedUid{0};
        if (!boost::conversion::try_lexical_convert(uid, convertedUid)) {
            return make_unexpected(error_code{Error::userNotFound});
        }

        if (newContacts.empty()) {
            return make_unexpected(error_code{Error::badRequest});
        }

        const auto provider(makeConnectionProvider(context, services::db::PassportUserId {convertedUid}));

        const auto transact = [&] (auto&& transaction) {
            auto createdContacts = contacts::createContacts(std::move(newContacts), transaction);

            if (!createdContacts) {
                return createdContacts;
            }

            return commit(provider, std::move(transaction))
                .bind([&] (auto&&) { return std::move(createdContacts); });
        };

        return begin(provider)
            .bind(transact);
    }

private:
    MakeConnectionProvider makeConnectionProvider;
};

} // namespace collie::logic::db
