#pragma once

#include "contacts/get_contacts.hpp"
#include "shared_contacts_impl.hpp"

#include <src/logic/interface/get_abook_format_contacts.hpp>
#include <src/services/db/contacts/query.hpp>
#include <src/services/db/passport_user_id.hpp>

#include <boost/range/algorithm/for_each.hpp>

namespace collie::logic::db {

template <class MakeConnectionProvider>
class GetAbookFormatContactsImpl final : public GetAbookFormatContacts {
public:
    GetAbookFormatContactsImpl(MakeConnectionProvider makeConnectionProvider)
            : makeConnectionProvider(makeConnectionProvider)
            , sharedContactsImpl(std::move(makeConnectionProvider)) {
    }

    virtual expected<services::abook::SearchContactsResult> operator()(
            const TaskContextPtr& context,
            const Uid& uid,
            const std::optional<std::string_view>& mixin,
            const std::optional<std::string_view>&) const override {
        std::int64_t numericUid;
        if (!boost::conversion::try_lexical_convert<std::int64_t>(uid, numericUid)) {
            LOGDOG_(context->logger(), error, log::uid=uid, log::message="failed to convert UID");
            return make_unexpected(error_code(Error::userNotFound));
        }

        auto provider{makeConnectionProvider(context, services::db::PassportUserId{numericUid})};

        auto getSharedContacts {[&] (auto&& existingContacts) -> expected<ExistingContacts> {
            if (mixin) {
                auto sharedContacts{sharedContactsImpl.getSharedContactsFromOrganisation(provider, context,
                        uid)};
                if (!sharedContacts) {
                    return make_unexpected(sharedContacts.error());
                } else {
                    existingContacts.contacts.insert(
                        existingContacts.contacts.end(),
                        std::make_move_iterator(sharedContacts->contacts.begin()),
                        std::make_move_iterator(sharedContacts->contacts.end())
                    );

                }
            }
            return existingContacts;
        }};

        auto makeResult{[&](auto&& contacts){return makeSearchContactsUngroupedResult(std::move(contacts));}};

        return contacts::getContacts<services::db::contacts::query::GetContacts>(provider, {}, {}, {}, {}).
                bind(std::move(getSharedContacts)).bind(std::move(makeResult));
    }

private:
    services::abook::SearchContactsResult makeSearchContactsUngroupedResult(
            ExistingContacts&& existingContacts) const {
        auto emailCount{0};
        boost::for_each(existingContacts.contacts, [&](const auto& contact){
                emailCount += contact.emails.size();});

        services::abook::SearchContactsUngroupedResult result;
        result.contact.reserve(emailCount);
        for (auto&& contact : existingContacts.contacts) {
            auto emails{makeAbookContactEmails(std::move(contact))};
            result.contact.insert(result.contact.end(), std::make_move_iterator(emails.begin()),
                    std::make_move_iterator(emails.end()));
        }

        result.count = result.contact.size();
        return result;
    }

    std::vector<services::abook::ContactEmail> makeAbookContactEmails(
            ExistingContact&& existingContact) const {
        std::vector<services::abook::ContactEmail> emails;
        const auto name{makeName(std::move(existingContact.vcard.names))};
        const auto photoPartialUrl{makePhotoPartialUrl(std::move(existingContact.vcard.photos))};
        const auto organization{getOrganization(std::move(existingContact.vcard.organizations))};
        const auto yaDirectory{makeYaDirectory(std::move(existingContact.vcard.directory_entries))};
        auto transformer{[&](auto&& existingEmail) {
            return services::abook::ContactEmail {
                existingContact.contact_id,
                std::move(existingEmail.id),
                std::move(existingEmail.tags),
                name,
                std::move(existingEmail.value),
                photoPartialUrl,
                organization.company,
                organization.department,
                organization.title,
                {},
                yaDirectory
            };
        }};

        std::transform(
                std::make_move_iterator(existingContact.emails.begin()),
                std::make_move_iterator(existingContact.emails.end()),
                std::back_inserter(emails),
                std::move(transformer));

        return emails;
    }

    Organization getOrganization(std::optional<std::vector<Organization>>&& organizations) const {
        return (organizations && !organizations->empty()) ? std::move((*organizations)[0]) : Organization{};
    }

    std::optional<services::abook::Name> makeName(std::optional<std::vector<Name>>&& names) const {
        if (!names || names->empty()) {
            return {};
        }

        auto& name{(*names)[0]};
        return services::abook::Name{std::move(name.first), std::move(name.middle), std::move(name.last)};
    }

    std::optional<std::string> makePhotoPartialUrl(std::optional<std::vector<Photo>>&& photos) const {
        return (photos && !photos->empty()) ? std::move((*photos)[0].uri) : std::optional<std::string>{};
    }

    std::optional<std::string> makeYaDirectoryType(std::optional<std::vector<std::string>> type) const {
        if (type && !type->empty()) {
            return {std::move((*type)[0])};
        }

        return {};
    }

    std::optional<services::abook::YaDirectory> makeYaDirectory(
            std::optional<std::vector<DirectoryEntry>>&& directoryEntries) const {
        if (!directoryEntries || directoryEntries->empty()) {
            return {};
        }

        auto& directoryEntry{(*directoryEntries)[0]};
        return services::abook::YaDirectory{
            std::move(directoryEntry.org_id),
            std::move(directoryEntry.org_name),
            makeYaDirectoryType(std::move(directoryEntry.type)),
            std::move(directoryEntry.entry_id)};
    }

    MakeConnectionProvider makeConnectionProvider;
    SharedContactsImpl<MakeConnectionProvider> sharedContactsImpl;
};

} // namespace collie::logic::db
