#pragma once

#include <src/logic/interface/get_changes.hpp>
#include <src/logic/interface/types/reflection/change.hpp>
#include <src/services/db/contacts/query.hpp>
#include <src/services/db/passport_user_id.hpp>
#include <src/services/db/request.hpp>

#include <boost/hana/core/to.hpp>
#include <boost/range/algorithm/transform.hpp>

namespace collie::logic::db {

template <class MakeConnectionProvider>
class GetChangesImpl final : public GetChanges {
public:
    explicit GetChangesImpl(MakeConnectionProvider makeConnectionProvider)
        : makeConnectionProvider(std::move(makeConnectionProvider)) {
    }

    virtual expected<Changes> operator()(const TaskContextPtr& context, const Uid& uid) const override {
        using services::db::PassportUserId;

        services::db::contacts::query::GetChanges q;
        if (!boost::conversion::try_lexical_convert<std::int64_t>(uid, q.uid)) {
            return make_unexpected(error_code(Error::userNotFound));
        }
        return services::db::request(
                services::db::retry(makeConnectionProvider(context, PassportUserId {q.uid})),
                std::as_const(q)
            ).bind([&](auto&& result) {
                Changes changes;
                changes.changes.reserve(result.size());
                boost::transform(result, std::back_inserter(changes.changes), [](const auto& changeRow) {
                    return Change{changeRow.revision, changeRow.change_date, makeChangeType(
                            changeRow.type.get())};
                });

                LOGDOG_(context->logger(), notice, log::query=ozo::get_query_name(q), log::uid=uid);

                return changes;
            });
    }

private:
    MakeConnectionProvider makeConnectionProvider;
};

} // namespace collie::logic::db
