#pragma once

#include "contacts/get_contacts_count.hpp"
#include "shared_contacts_impl.hpp"

#include <src/logic/interface/get_contacts_count.hpp>

#include <src/logic/error.hpp>
#include <src/services/db/contacts/query.hpp>
#include <src/services/db/passport_user_id.hpp>
#include <src/services/db/request.hpp>
#include <src/services/db/utils.hpp>

namespace collie::logic::db {

template <class MakeConnectionProvider>
class GetContactsCountImpl final : public GetContactsCount {
public:
    GetContactsCountImpl(MakeConnectionProvider makeConnectionProvider)
            : makeConnectionProvider(makeConnectionProvider)
            , sharedContactsImpl(std::move(makeConnectionProvider)) {
    }

    auto operator ()(
        const TaskContextPtr& context,
        const Uid& uid,
        const std::optional<std::string_view>& mixin,
        bool sharedWithEmails) const
            -> decltype(expected<ContactsCountersResult>()) override
    {
        std::int64_t numericUid;
        if (!boost::conversion::try_lexical_convert<std::int64_t>(uid, numericUid)) {
            LOGDOG_(context->logger(), error, log::uid=uid, log::message="failed to convert UID");
            return make_unexpected(error_code{Error::userNotFound});
        }

        auto provider{makeConnectionProvider(context, services::db::PassportUserId{numericUid})};
        auto getSharedContactsCount {[&] (auto&& contactsCountResult) -> expected<ContactsCountersResult> {
            if (mixin) {
                auto sharedContactsCount{sharedContactsImpl.getSharedContactsCountFromOrganisation(provider,
                        context, uid, sharedWithEmails)};
                if (!sharedContactsCount) {
                    return make_unexpected(sharedContactsCount.error());
                } else {
                    contactsCountResult.total += sharedContactsCount->total;
                    contactsCountResult.book.insert(
                        contactsCountResult.book.end(),
                        std::make_move_iterator(sharedContactsCount->book.begin()),
                        std::make_move_iterator(sharedContactsCount->book.end())
                    );
                }
            }

            return contactsCountResult;
        }};

        auto makeResult{[&](auto&& count){return ContactsCountersResult{count, {{uid, count}}};}};
        return contacts::getContactsCount<services::db::contacts::query::GetContactsCount>(provider, {}).
                bind(std::move(makeResult)).bind(std::move(getSharedContactsCount));
    }

private:
    MakeConnectionProvider makeConnectionProvider;
    SharedContactsImpl<MakeConnectionProvider> sharedContactsImpl;
};

} // namespace collie::logic::db
