#pragma once

#include "contacts/get_contacts.hpp"
#include "shared_contacts_impl.hpp"
#include "utils.hpp"

#include <src/logic/interface/get_contacts.hpp>
#include <src/services/db/contacts/query.hpp>
#include <src/services/db/passport_user_id.hpp>

#include <boost/range/algorithm_ext/erase.hpp>

namespace collie::logic::db {

template <class MakeConnectionProvider>
class GetContactsImpl final : public GetContacts {
public:
    GetContactsImpl(MakeConnectionProvider makeConnectionProvider)
            : makeConnectionProvider(makeConnectionProvider)
            , sharedContactsImpl(std::move(makeConnectionProvider)) {
    }

    virtual expected<ExistingContacts> operator()(
            const TaskContextPtr& context,
            const Uid& uid,
            const std::vector<ContactId>& contactIds,
            const std::optional<std::string_view>& mixin,
            const std::optional<std::string_view>& offset,
            const std::optional<std::string_view>& limit,
            bool onlyShared,
            bool sharedWithEmails) const override {
        std::int64_t numericUid;
        if (!boost::conversion::try_lexical_convert<std::int64_t>(uid, numericUid)) {
            LOGDOG_(context->logger(), error, log::uid=uid, log::message="failed to convert UID");
            return make_unexpected(error_code(Error::userNotFound));
        }

        std::optional<std::int64_t> numericOffset;
        std::optional<std::int64_t> numericLimit;
        const std::string offsetName{"offset"};
        const std::string limitName{"limit"};
        if (!convertToNonnegativeNumber(context->logger(), offset, numericOffset, offsetName) ||
            !convertToNonnegativeNumber(context->logger(), limit, numericLimit, limitName)) {
            return make_unexpected(error_code(Error::badRequest));
        }

        auto provider{makeConnectionProvider(context, services::db::PassportUserId{numericUid})};
        auto addSharedContacts{[&](auto&& existingContacts) {
            if (mixin || onlyShared) {
                auto sharedContacts{sharedContactsImpl.getSharedContactsFromOrganisation(provider, context,
                        uid)};
                if (!sharedContacts) {
                    return make_expected_from_error<ExistingContacts>(std::move(sharedContacts).error());
                }

                auto actualSharedContacts{postprocessSharedContacts(std::move(sharedContacts->contacts),
                        numericOffset, numericLimit, sharedWithEmails)};
                existingContacts.contacts.insert(
                        existingContacts.contacts.end(),
                        std::make_move_iterator(actualSharedContacts.begin()),
                        std::make_move_iterator(actualSharedContacts.end()));
            }

            return make_expected(existingContacts);
        }};

        if (onlyShared) {
            return addSharedContacts(ExistingContacts{});
        }

        return contacts::getContacts<services::db::contacts::query::GetContacts>(provider, contactIds, {},
                numericOffset, numericLimit).bind(std::move(addSharedContacts));
    }

private:
    std::vector<ExistingContact> postprocessSharedContacts(
            std::vector<ExistingContact> contacts,
            const std::optional<std::int64_t>& offset,
            const std::optional<std::int64_t>& limit,
            bool sharedWithEmails) const {
        if (sharedWithEmails) {
            boost::remove_erase_if(contacts, [](const auto& contact){return !hasAtLeastOneEmail(
                    contact.emails);});
        }

        auto begin{contacts.begin()};
        auto end{contacts.end()};
        if (offset) {
            begin = (*offset <= std::distance(begin, end)) ? (begin + *offset) : end;
        }

        if (limit && (*limit < std::distance(begin, end))) {
            end = begin + *limit;
        }

        return {std::make_move_iterator(begin), std::make_move_iterator(end)};
    }

    MakeConnectionProvider makeConnectionProvider;
    SharedContactsImpl<MakeConnectionProvider> sharedContactsImpl;
};

} // namespace collie::logic::db
