#pragma once

#include "contacts/get_contacts_by_tag_id.hpp"
#include "utils.hpp"

#include <src/logic/interface/get_contacts_with_tag.hpp>
#include <src/services/db/passport_user_id.hpp>

namespace collie::logic::db {

template <class MakeConnectionProvider>
class GetContactsWithTagImpl final : public GetContactsWithTag {
public:
    GetContactsWithTagImpl(MakeConnectionProvider makeConnectionProvider)
            : makeConnectionProvider(std::move(makeConnectionProvider)) {
    }

    virtual expected<ExistingContacts> operator()(
            const TaskContextPtr& context,
            const Uid& uid,
            TagId tagId,
            const std::optional<std::string_view>& offset,
            const std::optional<std::string_view>& limit) const override {
        std::int64_t numericUid;
        if (!boost::conversion::try_lexical_convert<std::int64_t>(uid, numericUid)) {
            LOGDOG_(context->logger(), error, log::uid=uid, log::message="failed to convert UID");
            return make_unexpected(error_code(Error::userNotFound));
        }

        std::optional<std::int64_t> numericOffset;
        std::optional<std::int64_t> numericLimit;
        const std::string offsetName{"offset"};
        const std::string limitName{"limit"};
        if (!convertToNonnegativeNumber(context->logger(), offset, numericOffset, offsetName) ||
                !convertToNonnegativeNumber(context->logger(), limit, numericLimit, limitName)) {
            return make_unexpected(error_code(Error::badRequest));
        }

        auto provider{services::db::retry(makeConnectionProvider(context, services::db::PassportUserId{
                numericUid}))};
        return contacts::getContactsByTagId(std::move(provider), tagId, std::move(numericOffset), std::move(
                numericLimit));
    }

private:
    MakeConnectionProvider makeConnectionProvider;
};

} // namespace collie::logic::db
