#pragma once

#include "contacts/get_contacts_count.hpp"
#include "contacts/get_subscribed_list.hpp"

#include <src/logic/interface/get_shared_contacts_count_from_list.hpp>
#include <src/services/db/contacts/query.hpp>
#include <src/services/db/org_user_id.hpp>
#include <src/services/db/passport_user_id.hpp>

#include <boost/lexical_cast/try_lexical_convert.hpp>

namespace collie::logic::db {

template <class MakeConnectionProvider>
class GetSharedContactsCountFromListImpl final : public GetSharedContactsCountFromList {
public:
    GetSharedContactsCountFromListImpl(MakeConnectionProvider makeConnectionProvider)
        : makeConnectionProvider(makeConnectionProvider) {
    }

    virtual expected<ListContactsCounter> operator()(const TaskContextPtr& context, const Uid& uid,
            ListId listId, bool sharedWithEmails) const override {
        std::int64_t numericUid;
        if (!boost::conversion::try_lexical_convert<std::int64_t>(uid, numericUid)) {
            return make_unexpected(error_code{Error::userNotFound});
        }

        using services::db::PassportUserId;
        using services::db::OrgUserId;
        auto userProvider{makeConnectionProvider(context, services::db::PassportUserId{numericUid})};
        auto getContactsCount{[&](const auto& subscribedList) {
            if (!subscribedList)  {
                const std::string errorMessage{"list_id " + std::to_string(listId) + " not found"};
                LOGDOG_(context->logger(), error, log::uid=uid, log::message=errorMessage);
                return make_expected_from_error<std::int64_t>(error_code{Error::badRequest});
            }

            auto sharedProvider{(subscribedList->owner_user_type == "connect_organization")
                ? makeConnectionProvider(context, OrgUserId{subscribedList->owner_user_id})
                : makeConnectionProvider(context, PassportUserId{subscribedList->owner_user_id})};

            if (sharedWithEmails) {
                using services::db::contacts::query::GetContactsWithEmailsCount;
                return contacts::getContactsCount<GetContactsWithEmailsCount>(std::move(sharedProvider),
                        {subscribedList->owner_list_id});
            } else {
                using services::db::contacts::query::GetContactsCount;
                return contacts::getContactsCount<GetContactsCount>(std::move(sharedProvider),
                        {subscribedList->owner_list_id});
            }
        }};

        auto makeResult{[&](auto&& count){return ListContactsCounter{count};}};
        return contacts::getSubscribedList(std::move(userProvider), listId).bind(std::move(getContactsCount)).
                bind(std::move(makeResult));
    }

private:
    MakeConnectionProvider makeConnectionProvider;
};

} // namespace collie::logic::db
