#pragma once

#include "contacts/get_contacts.hpp"
#include "contacts/get_subscribed_list.hpp"
#include "utils.hpp"

#include <src/logic/interface/get_shared_contacts_from_list.hpp>
#include <src/services/db/contacts/query.hpp>
#include <src/services/db/org_user_id.hpp>
#include <src/services/db/passport_user_id.hpp>

#include <boost/lexical_cast/try_lexical_convert.hpp>

namespace collie::logic::db {

template<typename MakeConnectionProvider>
class GetSharedContactsFromListImpl final : public GetSharedContactsFromList {
public:
    GetSharedContactsFromListImpl(MakeConnectionProvider makeConnectionProvider)
        : makeConnectionProvider(std::move(makeConnectionProvider)) {
    }

    virtual expected<ExistingContacts> operator()(
            const TaskContextPtr& context,
            const Uid& uid,
            const std::vector<ContactId>& contactIds,
            ListId listId,
            const std::optional<std::string_view>& offset,
            const std::optional<std::string_view>& limit,
            bool sharedWithEmails) const override {
        std::int64_t numericUid;
        if (!boost::conversion::try_lexical_convert<decltype(numericUid)>(uid, numericUid)) {
            LOGDOG_(context->logger(), error, log::uid=uid, log::message="failed to convert UID");
            return make_unexpected(error_code{Error::userNotFound});
        }

        std::optional<std::int64_t> numericOffset;
        std::optional<std::int64_t> numericLimit;
        const std::string offsetName{"offset"};
        const std::string limitName{"limit"};
        if (!convertToNonnegativeNumber(context->logger(), offset, numericOffset, offsetName) ||
            !convertToNonnegativeNumber(context->logger(), limit, numericLimit, limitName)) {
            return make_unexpected(error_code{Error::badRequest});
        }

        using services::db::PassportUserId;
        using services::db::OrgUserId;
        auto userProvider{makeConnectionProvider(context, PassportUserId{numericUid})};
        auto getContacts{[&](const auto& subscribedList) {
            if (!subscribedList)  {
                const std::string errorMessage{"list_id " + std::to_string(listId) + " not found"};
                LOGDOG_(context->logger(), error, log::uid=uid, log::message=errorMessage);
                return make_expected_from_error<ExistingContacts>(error_code{Error::badRequest});
            }

            auto sharedProvider{(subscribedList->owner_user_type == "connect_organization")
                ? makeConnectionProvider(context, OrgUserId{subscribedList->owner_user_id})
                : makeConnectionProvider(context, PassportUserId{subscribedList->owner_user_id})};

            if (sharedWithEmails) {
                using services::db::contacts::query::GetContactsWithEmails;
                return contacts::getContacts<GetContactsWithEmails>(std::move(sharedProvider), contactIds,
                        {subscribedList->owner_list_id}, numericOffset, numericLimit);
            } else {
                using services::db::contacts::query::GetContacts;
                return contacts::getContacts<GetContacts>(std::move(sharedProvider), contactIds,
                        {subscribedList->owner_list_id}, numericOffset, numericLimit);
            }
        }};

        return contacts::getSubscribedList(std::move(userProvider), listId).bind(std::move(getContacts));
    }

private:
    MakeConnectionProvider makeConnectionProvider;
};

} // namespace collie::logic::db
