#pragma once

#include <src/error_code.hpp>
#include <src/expected.hpp>
#include <src/logic/interface/get_shared_lists.hpp>
#include <src/logic/db/contacts/get_user_type_lists.hpp>
#include <src/services/db/connection_provider.hpp>
#include <src/services/db/passport_user_id.hpp>
#include <boost/lexical_cast/try_lexical_convert.hpp>

namespace collie::logic::db {

template <class MakeConnectionProvider>
class GetSharedListsImpl final : public GetSharedLists {
public:
    GetSharedListsImpl(MakeConnectionProvider makeConnectionProvider)
        : makeConnectionProvider(std::move(makeConnectionProvider)) {
    }

    virtual expected<ExistingSharedLists> operator()(const TaskContextPtr& context, const Uid& uid) const override {
        std::int64_t numericUid;
        if (!boost::conversion::try_lexical_convert<std::int64_t>(uid, numericUid)) {
            return make_unexpected(error_code(Error::userNotFound));
        }

        auto provider{makeConnectionProvider(context, services::db::PassportUserId{numericUid})};
        return contacts::getSubscribedLists(provider).bind([&](auto&& subscribedLists) {
            if (subscribedLists.empty()) {
                return make_expected(ExistingSharedLists{});
            }

            return contacts::getUserTypeLists(provider);
        });
    }

private:
    MakeConnectionProvider makeConnectionProvider;
};

} // namespace collie::logic::db
