#pragma once

#include <src/logic/interface/get_tags.hpp>
#include <src/logic/interface/types/reflection/existing_tag.hpp>
#include <src/logic/error.hpp>
#include <src/services/db/contacts/query.hpp>
#include <src/services/db/passport_user_id.hpp>
#include <src/services/db/request.hpp>
#include <src/services/db/utils.hpp>

#include <boost/hana/core/to.hpp>
#include <boost/range/algorithm/transform.hpp>

namespace collie::logic::db {

template <class MakeConnectionProvider>
class GetTagsImpl final : public GetTags {
public:
    GetTagsImpl(MakeConnectionProvider makeConnectionProvider)
            : makeConnectionProvider(std::move(makeConnectionProvider)) {
    }

    virtual expected<ExistingTags> operator()(const TaskContextPtr& context, const Uid& uid) const override {
        using services::db::PassportUserId;

        services::db::contacts::query::GetTags q;
        if (!boost::conversion::try_lexical_convert<std::int64_t>(uid, q.uid)) {
            return make_unexpected(error_code(Error::userNotFound));
        }

        return services::db::request(
                services::db::retry(makeConnectionProvider(context, PassportUserId {q.uid})),
                std::as_const(q)
            ).bind([&] (auto&& result) {
                ExistingTags tags;
                tags.tags.reserve(result.size());
                boost::transform(result, std::back_inserter(tags.tags), [](auto& row) {
                    return ExistingTag {
                        row.tag_id,
                        makeTagType(row.type.get()),
                        row.name,
                        row.contacts_count,
                        row.revision
                    };
                });

                LOGDOG_(context->logger(), notice,
                        log::query=ozo::get_query_name(q),
                        log::uid=uid
                );
                return tags;
            });
    }

private:
    MakeConnectionProvider makeConnectionProvider;
};

} // namespace collie::logic::db
