#pragma once

#include "contacts/remove_contacts_completely.hpp"

#include <src/logic/interface/remove_contacts.hpp>
#include <src/services/db/begin.hpp>
#include <src/services/db/commit.hpp>
#include <src/services/db/passport_user_id.hpp>

namespace collie::logic::db {

template <class MakeConnectionProvider>
class RemoveContactsImpl final : public RemoveContacts {
public:
    explicit RemoveContactsImpl(MakeConnectionProvider makeConnectionProvider)
        : makeConnectionProvider(std::move(makeConnectionProvider)) {
    }

    expected<Revision> operator()(const TaskContextPtr& context, const Uid& uid,
            const std::vector<ContactId>& contactIds) const override {
        using services::db::PassportUserId;
        std::int64_t numericUid;
        if (!boost::conversion::try_lexical_convert<std::int64_t>(uid, numericUid)) {
            return make_unexpected(error_code(Error::userNotFound));
        }

        auto provider{makeConnectionProvider(context, services::db::PassportUserId{numericUid})};
        const auto transact{[&](auto&& transaction) {
            const auto commit{[&](auto&& revision){return services::db::commit(std::move(provider),
                    std::move(transaction)).bind([&](auto&&){return revision;});}};
            return contacts::removeContactsCompletely(transaction, contactIds).bind(std::move(commit));
        }};

        return services::db::begin(provider).bind(std::move(transact));
    }

private:
    MakeConnectionProvider makeConnectionProvider;
};

} // namespace collie::logic::db
