#pragma once

#include "contacts/get_contact_ids_by_tag_ids.hpp"
#include "contacts/get_contacts_email_ids_by_tag_ids.hpp"
#include "contacts/remove_tag.hpp"
#include "contacts/untag_contacts.hpp"
#include "contacts/untag_contacts_emails.hpp"

#include <src/logic/interface/remove_tag.hpp>
#include <src/services/db/begin.hpp>
#include <src/services/db/commit.hpp>
#include <src/services/db/passport_user_id.hpp>

namespace collie::logic::db {

template <class MakeConnectionProvider>
class RemoveTagImpl final : public RemoveTag {
public:
    explicit RemoveTagImpl(MakeConnectionProvider makeConnectionProvider)
        : makeConnectionProvider(std::move(makeConnectionProvider)) {
    }

    expected<Revision> operator()(const TaskContextPtr& context, const Uid& uid,
            const TagId tagId) const override {
        using services::db::PassportUserId;
        std::int64_t numericUid;
        if (!boost::conversion::try_lexical_convert<std::int64_t>(uid, numericUid)) {
            return make_unexpected(error_code(Error::userNotFound));
        }

        auto provider{makeConnectionProvider(context, services::db::PassportUserId{numericUid})};
        const auto transact{[&](auto&& transaction) {
            const auto untagContactsEmails{[&](auto&& emailIds) {
                if (emailIds.empty()) {
                    return make_expected(Revision{});
                }

                return contacts::untagContactsEmails(transaction, tagId, std::move(emailIds));
            }};

            const auto getContactIdsByTagIds{[&](auto&&){return contacts::getContactIdsByTagIds(
                    transaction, {tagId});}};
            const auto untagContacts{[&](auto&& contactIds) {
                if (contactIds.empty()) {
                    return make_expected(Revision{});
                }

                return contacts::untagContacts(transaction, tagId, std::move(contactIds));
            }};

            const auto removeTag{[&](auto&&){return contacts::removeTag(transaction, tagId);}};
            const auto commit{[&](auto&& revision){return services::db::commit(std::move(provider),
                    std::move(transaction)).bind([&](auto&&){return revision;});}};
            return contacts::getContactsEmailIdsByTagIds(transaction, {tagId}).
                    bind(std::move(untagContactsEmails)).
                    bind(std::move(getContactIdsByTagIds)).
                    bind(std::move(untagContacts)).
                    bind(std::move(removeTag)).
                    bind(std::move(commit));
        }};

        return services::db::begin(provider).bind(std::move(transact));
    }

private:
    MakeConnectionProvider makeConnectionProvider;
};

} // namespace collie::logic::db
