#pragma once

#include "contacts/get_shared_contacts.hpp"
#include "contacts/get_shared_contacts_count.hpp"
#include "contacts/get_shared_contacts_count_with_emails.hpp"
#include "contacts/get_subscribed_lists.hpp"
#include "contacts/subscribe_to_contacts.hpp"

#include <src/expected.hpp>
#include <src/logic/interface/types/contacts_counters_result.hpp>
#include <src/logic/interface/types/existing_contacts.hpp>
#include <src/logic/interface/types/uid.hpp>
#include <src/services/db/org_user_id.hpp>
#include <src/task_context.hpp>

#include <boost/range/adaptor/filtered.hpp>

namespace collie::logic::db {

using SubscribedLists = std::vector<services::db::contacts::SubscribeList>;

template <class MakeConnectionProvider>
class SharedContactsImpl {
public:

    SharedContactsImpl(MakeConnectionProvider makeConnectionProvider)
        : makeConnectionProvider(std::move(makeConnectionProvider)) {
    }

    template<typename Connection> expected<ExistingContacts> getSharedContactsFromOrganisation(
        Connection&& userProvider,
        const TaskContextPtr& context,
        const Uid& uid
    ) const {
        return contacts::getSubscribedLists(userProvider)
            .bind([&] (auto&& subscribedLists) -> expected<ExistingContacts> {
                if (subscribedLists.empty()) {
                    return ExistingContacts {};
                }
                return getSharedContacts(context, uid, "connect_organization", std::move(subscribedLists));
            });
    }

    template<typename Connection> expected<ContactsCountersResult> getSharedContactsCountFromOrganisation(
        Connection&& userProvider,
        const TaskContextPtr& context,
        const Uid& uid,
        bool sharedWithEmails
    ) const {
        return contacts::getSubscribedLists(userProvider)
            .bind([&] (auto&& subscribedLists) -> expected<ContactsCountersResult> {
                if (subscribedLists.empty()) {
                    ContactsCountersResult contactsCountersResult;
                    contactsCountersResult.total = 0;
                    return contactsCountersResult;
                }
                return getSharedContactsCount(context, uid, "connect_organization", std::move(subscribedLists), sharedWithEmails);
            });
    }

    expected<ExistingContacts> getSharedContacts(
        const TaskContextPtr& context,
        const Uid& uid,
        const std::string& userType,
        SubscribedLists subscribedLists
    ) const {
        ExistingContacts sharedContacts;

        std::int64_t numericUid;
        if (!boost::conversion::try_lexical_convert<std::int64_t>(uid, numericUid)) {
            LOGDOG_(context->logger(), error, log::uid = uid, log::message = "failed to convert UID");
            return make_unexpected(error_code(Error::userNotFound));
        }

        auto listIdsByUserId = transformSubscribedLists(userType, std::move(subscribedLists));

        for (auto&& [key, list_ids]: listIdsByUserId) {
            auto orgProvider {services::db::retry(makeConnectionProvider(context, services::db::OrgUserId {
                std::move(key)}))};

            auto result = contacts::getSharedContacts(
                std::move(orgProvider),
                numericUid,
                "passport_user",
                std::move(list_ids)
            );
            if (!result) {
                return make_unexpected(error_code(result.error()));
            }
            sharedContacts.contacts.insert(
                sharedContacts.contacts.end(),
                std::make_move_iterator(result->contacts.begin()),
                std::make_move_iterator(result->contacts.end())
            );
        }
        return sharedContacts;
    }

    expected<ContactsCountersResult> getSharedContactsCount(
        const TaskContextPtr& context,
        const Uid& uid,
        const std::string& userType,
        SubscribedLists subscribedLists,
        bool sharedWithEmails
    ) const {
        ContactsCountersResult contactsCountersResult;
        contactsCountersResult.total = 0;

        std::int64_t numericUid;
        if (!boost::conversion::try_lexical_convert<std::int64_t>(uid, numericUid)) {
            LOGDOG_(context->logger(), error, log::uid = uid, log::message = "failed to convert UID");
            return make_unexpected(error_code(Error::userNotFound));
        }

        auto listIdsByUserId = transformSubscribedLists(userType, std::move(subscribedLists));

        for (auto&& [key, list_ids]: listIdsByUserId) {
            auto orgProvider {services::db::retry(makeConnectionProvider(context, services::db::OrgUserId {
                key}))};

            auto result = expected<std::int64_t>();
            if (sharedWithEmails) {
                result = contacts::getSharedContactsCountWithEmails(
                    std::move(orgProvider),
                    numericUid,
                    "passport_user",
                    std::move(list_ids));
            } else {
                result = contacts::getSharedContactsCount(
                    std::move(orgProvider),
                    numericUid,
                    "passport_user",
                    std::move(list_ids));
            }
            if (!result) {
                return make_unexpected(error_code(result.error()));
            }
            std::string id;
            if (!boost::conversion::try_lexical_convert<std::string>(key, id)) {
                LOGDOG_(context->logger(), error, log::uid = uid, log::message = "failed to convert UID");
                return make_unexpected(error_code(Error::badRequest));
            }

            contactsCountersResult.total += result.value();
            contactsCountersResult.book.emplace_back(ContactCounter{std::move(id), std::move(result).value()});
        }
        return contactsCountersResult;
    }

    std::unordered_map<std::int64_t, std::vector<std::int64_t>> transformSubscribedLists(
            const std::string& userType, SubscribedLists subscribedLists) const {
        const auto filterByUserType{boost::adaptors::filtered([&](const auto& list){
                return list.owner_user_type == userType;})};
        std::unordered_map<std::int64_t, std::vector<std::int64_t>> listIdsByUserId;
        for (auto&& subscribeList: subscribedLists | filterByUserType) {
            const auto it{listIdsByUserId.find(subscribeList.owner_user_id)};
            if (it != listIdsByUserId.end()) {
                it->second.push_back(std::move(subscribeList.owner_list_id));
            } else {
                listIdsByUserId.emplace(subscribeList.owner_user_id, std::vector<ListId>{subscribeList.
                        owner_list_id});
            }
        };

        return listIdsByUserId;
    }

private:
    MakeConnectionProvider makeConnectionProvider;

};

}
