#pragma once

#include "contacts/acquire_revision.hpp"
#include "contacts/update_contacts.hpp"

#include <src/logic/interface/update_contacts.hpp>
#include <src/services/db/contacts/query.hpp>
#include <src/services/db/passport_user_id.hpp>
#include <src/services/db/begin.hpp>
#include <src/services/db/commit.hpp>

namespace collie::logic::db {

template <class MakeConnectionProvider>
class UpdateContactsImpl final : public UpdateContacts {
public:
    explicit UpdateContactsImpl(MakeConnectionProvider makeConnectionProvider)
        : makeConnectionProvider(std::move(makeConnectionProvider)) {
    }

    expected<Revision> operator ()(const TaskContextPtr& context, const Uid& uid,
            UpdatedContacts updatedContacts) const override {
        using services::db::PassportUserId;
        using services::db::begin;
        using services::db::commit;

        if (updatedContacts.updated_contacts.empty()) {
            return expected(Revision{});
        }

        std::int64_t numericUid;
        if (!boost::conversion::try_lexical_convert<std::int64_t>(uid, numericUid)) {
            return make_unexpected(error_code(Error::userNotFound));
        }

        const auto provider = makeConnectionProvider(context, PassportUserId {numericUid});

        const auto update = [&] (auto&& transaction) {
            const auto revision = contacts::acquireRevision(transaction)
                .bind([&] (auto&&) {
                    return contacts::updateContacts(std::move(updatedContacts), transaction);
                });

            if (!revision) {
                return revision;
            }

            const auto returnRevision = [&] (auto&&) { return revision.value(); };

            return commit(provider, std::move(transaction))
                .bind(returnRevision);
        };

        return begin(provider)
            .bind(update);
    }

private:
    MakeConnectionProvider makeConnectionProvider;
};

} // namespace collie::logic::db
