#pragma once

#include "contacts/get_tag_by_id.hpp"
#include "contacts/tag_contacts_and_contacts_emails.hpp"
#include "contacts/untag_contacts_and_contacts_emails.hpp"
#include "contacts/update_tag.hpp"

#include <src/logic/interface/update_tag.hpp>
#include <src/services/db/begin.hpp>
#include <src/services/db/commit.hpp>
#include <src/services/db/passport_user_id.hpp>

namespace collie::logic::db {

template <class MakeConnectionProvider>
class UpdateTagImpl final : public UpdateTag {
public:
    explicit UpdateTagImpl(MakeConnectionProvider makeConnectionProvider)
        : makeConnectionProvider(std::move(makeConnectionProvider)) {
    }

    expected<Revision> operator()(const TaskContextPtr& context, const Uid& uid, const TagId tagId,
            const UpdatedTag& updatedTag) const override {
        using services::db::PassportUserId;
        using services::db::begin;
        using services::db::commit;

        std::int64_t convertedUid{0};
        if (!boost::conversion::try_lexical_convert<std::int64_t>(uid, convertedUid)) {
            return make_unexpected(error_code(Error::userNotFound));
        }

        const auto provider(makeConnectionProvider(context, services::db::PassportUserId {convertedUid}));
        const auto transact = [&] (auto&& transaction) {
            std::optional<expected<Revision>> revision;
            if (updatedTag.name) {
                revision = contacts::updateTag(transaction, tagId, updatedTag);
                if (!(*revision)) {
                    return *revision;
                }
            }

            if (updatedTag.add_contact_ids || updatedTag.add_contact_email_ids) {
                revision = contacts::tagContactsAndContactsEmails(
                    transaction,
                    tagId,
                    updatedTag.add_contact_ids
                        ? std::move(*updatedTag.add_contact_ids)
                        : std::vector<ContactId>{},
                    updatedTag.add_contact_email_ids
                        ? std::move(*updatedTag.add_contact_email_ids)
                        : std::vector<ContactsEmailId>{});
                if (!(*revision)) {
                    return *revision;
                }
            }

            if (updatedTag.remove_contact_ids || updatedTag.remove_contact_email_ids) {
                revision = contacts::untagContactsAndContactsEmails(
                    transaction,
                    tagId,
                    updatedTag.remove_contact_ids
                        ? std::move(*updatedTag.remove_contact_ids)
                        : std::vector<ContactId>{},
                    updatedTag.remove_contact_email_ids
                        ? std::move(*updatedTag.remove_contact_email_ids)
                        : std::vector<ContactsEmailId>{});
                if (!(*revision)) {
                    return *revision;
                }
            }

            if (!revision) {
                auto existingTag{contacts::getTagById(transaction, tagId)};
                if (!existingTag) {
                    return make_expected_from_error<Revision>(std::move(existingTag).error());
                }

                return make_expected(existingTag->revision);
            }

            return commit(provider, std::move(transaction)).bind([&](auto&&){return *std::move(revision);});
        };

        return begin(provider).bind(transact);
    }

private:
    MakeConnectionProvider makeConnectionProvider;
};

} // namespace collie::logic::db
