#pragma once

#include <src/error_code.hpp>

namespace collie::logic {

enum class Error {
    ok,
    userNotFound,
    invalidUid,
    invalidDatabaseResult,
    badRequest
};

} // namespace collie::logic


namespace boost::system {

template <>
struct is_error_code_enum<collie::logic::Error> : std::true_type {};

} // namespace boost::system

namespace collie::logic {

class ErrorCategory final : public boost::system::error_category {
public:
    const char* name() const noexcept override {
        return "collie::logic::ErrorCategory";
    }

    std::string message(int value) const override {
        switch (static_cast<Error>(value)) {
            case Error::ok:
                return "ok";
            case Error::userNotFound:
                return "user is not found";
            case Error::invalidUid:
                return "uid has invalid format";
            case Error::invalidDatabaseResult:
                return "invalid database result";
            case Error::badRequest:
                return "bad request";
        }
        return "unknown error code: " + std::to_string(value);
    }

    static const ErrorCategory& instance() {
        static const ErrorCategory errorCategory;
        return errorCategory;
    }
};

inline error_code::base_type make_error_code(Error ec) {
    return error_code::base_type(static_cast<int>(ec), ErrorCategory::instance());
}

} // namespace collie::logic
