#include "abook_colabook_feed_addrdb_handler.hpp"

#include <src/log.hpp>
#include <src/logic/interface/types/reflection/created_contacts.hpp>
#include <src/server/respond.hpp>
#include <src/server/utils.hpp>

#include <butil/email/helpers.h>

#include <boost/range/adaptor/transformed.hpp>
#include <boost/range/adaptor/filtered.hpp>

namespace collie::server {

AbookColabookFeedAddrdbHandler::AbookColabookFeedAddrdbHandler(
        const std::shared_ptr<const logic::AddEmails>& impl)
    : impl(impl) {
}

expected<void> AbookColabookFeedAddrdbHandler::operator()(const StreamPtr& stream,
        const TaskContextPtr& context) const {
    using namespace std::string_literals;
    const auto& request = stream->request();
    const auto& params{request->url.params};
    const std::string uidKey{"uid"};
    const auto uidOpt{getOrNone(params, uidKey)};
    const std::string toKey{"to"};
    const auto toOpt{getOrNone(params, toKey)};
    if (!uidOpt || !toOpt) {
        LOGDOG_(context->logger(), error, log::message=R"(")" + ((uidOpt) ? toKey : uidKey) +
                R"(" not present in query string)");
        return make_unexpected(error_code(Error::invalidParameter));
    }
    const auto uid{std::string{*uidOpt}};
    const auto to{std::string{*toOpt}};
    const auto parsedEmails{EmailHelpers::toEmailVec(to)};
    const auto emails{ parsedEmails
        | boost::adaptors::filtered([](const auto& email) { return email.local().size() && email.domain().size();})
        | boost::adaptors::transformed([](const auto& email) { return EmailHelpers::toString(email);})};

    if (emails.empty()) {
        LOGDOG_(context->logger(), warning, log::uid=uid, log::message="empty valid emails from: "s + to);
        return respondWithJson(stream, logic::CreatedContacts{});
    }
    logic::Recipients recipients{std::vector<std::string>{std::make_move_iterator(emails.begin()),
        std::make_move_iterator(emails.end())}, {}, {}};
    return respondWithJson(stream, (*impl)(context, uid, std::move(recipients)));
}

} // namespace collie::server
