#include "carddav_multiget_handler.hpp"

#include <src/logic/interface/types/reflection/carddav_multiget_result.hpp>
#include <src/server/respond.hpp>
#include <src/utils.hpp>

#include <boost/range/algorithm/transform.hpp>

namespace collie::server {

CarddavMultigetHandler::CarddavMultigetHandler(const std::shared_ptr<const logic::CarddavMultiget>& impl)
    : impl(impl) {
}

expected<void> CarddavMultigetHandler::operator()(const Uid& uid, const StreamPtr& stream,
        const TaskContextPtr& context) const {
    const auto getUrisFromRequestBody{[&] {
        using Result = std::vector<std::string>;
        const auto& request = stream->request();
        const std::string body{request->raw_body.begin(), request->raw_body.end()};

        try {
            const std::string rootTagName{"request"};
            Result uris;
            boost::range::transform(ptreeFromXml(body).get_child(rootTagName), std::back_inserter(uris),
                    [](const auto& contact){return contact.second.template get_value<std::string>();});
            return yamail::make_expected(uris);
        } catch (const std::exception& ex) {
            using namespace std::string_literals;
            const std::string errorMessage{"failed to parse addressbook-multiget request body ("s +
                    ex.what() + ")"};
            logException(context->logger(), ex, log::message=errorMessage, log::request_body=body);
            return yamail::make_expected_from_error<Result>(error_code{
                    make_error_code(Error::invalidRequestBodyFormat), errorMessage});
        }
    }};

    const auto multiget{[&](auto&& uris) {
        auto result{(*impl)(context, uid.value, std::move(uris))};
        const std::string rootTagName{"multiget-result"};
        std::string attribute;
        if (result) {
            attribute = R"(found=")" + std::to_string(result.value().contact.size()) + R"(")";
        }

        return respondWithXml(stream, std::move(result), rootTagName, attribute);
    }};

    return getUrisFromRequestBody().bind(multiget);
}

} // namespace collie::server
