#pragma once

#include <src/server/error_category.hpp>
#include <src/expected.hpp>
#include <src/error_code.hpp>

#include <yamail/data/deserialization/yajl.h>

#include <boost/hana/string.hpp>
#include <boost/lexical_cast.hpp>

#include <cstdint>
#include <string>
#include <variant>
#include <vector>

namespace collie::server {

using namespace boost::hana::literals;

template <class Tag>
struct StringParameter {
    std::string value;

    static auto make(const std::string& value) {
        return StringParameter{value};
    }
};

template <class Value, class Tag>
struct JsonParameter {
    Value value;

    static std::variant<JsonParameter, expected<void>> make(const std::string& value) {
        using namespace std::string_literals;
        using yamail::data::deserialization::fromJson;
        try {
            return JsonParameter {fromJson<Value>(value)};
        } catch (const std::exception& e) {
            return make_expected_from_error<void>(error_code(
                Error::invalidParameter,
                boost::hana::to<const char*>(BOOST_HANA_STRING("failed to parse parameter ") + Tag {})
                + ": "s + e.what()
            ));
        }
    }
};

template <class Value, class Tag>
struct LexicalCastParameter {
    Value value;

    static std::variant<LexicalCastParameter, expected<void>> make(const std::string& value) {
        Value result;
        if (boost::conversion::try_lexical_convert(value, result)) {
            return LexicalCastParameter {std::move(result)};
        }
        return make_expected_from_error<void>(error_code(
            Error::invalidParameter,
            boost::hana::to<const char*>(BOOST_HANA_STRING("failed to parse parameter ") + Tag {})
        ));
    }
};

using Uid = StringParameter<decltype("uid"_s)>;
using Uri = StringParameter<decltype("uri"_s)>;
using Etag = StringParameter<decltype("etag"_s)>;
using ContactIds = JsonParameter<std::vector<std::int64_t>, decltype("contact_ids"_s)>;
using TagId = LexicalCastParameter<std::int64_t, decltype("tag_id"_s)>;
using TagIds = JsonParameter<std::vector<std::int64_t>, decltype("tag_ids"_s)>;
using ListId = LexicalCastParameter<std::int64_t, decltype("list_id"_s)>;
using Revision = LexicalCastParameter<std::int64_t, decltype("revision"_s)>;

} // namespace collie::server
