#pragma once

#include "parameters.hpp"

#include <src/server/error_category.hpp>
#include <src/server/utils.hpp>

#include <tvm_guard/log_tskv.h>

#include <boost/algorithm/string/case_conv.hpp>
#include <boost/hana/front.hpp>
#include <boost/hana/tuple.hpp>

#include <memory>

namespace collie::server {

namespace hana = boost::hana;

template <class TvmGuard>
class TvmGuarded {
public:
    TvmGuarded(std::shared_ptr<const TvmGuard> tvmGuard, std::string path)
            : tvmGuard(std::move(tvmGuard)), path(std::move(path)) {
        boost::to_lower_copy(std::back_inserter(serviceTicketHeader), tvm_guard::header::serviceTicket());
        boost::to_lower_copy(std::back_inserter(userTicketHeader), tvm_guard::header::userTicket());
    }

private:
    template <typename Continuation, typename... Parameters>
    auto check(Continuation&& continuation, hana::tuple<Parameters...>&& parameters, const std::string& uid,
            const StreamPtr& stream, const TaskContextPtr& context) const
            -> decltype(std::forward<Continuation>(continuation)(std::move(parameters), stream, context)) {
        const auto& request = stream->request();
        const auto result = tvmGuard->check(path, uid, getOrNone(request->headers, serviceTicketHeader),
            getOrNone(request->headers, userTicketHeader));
        LOGDOG_(context->logger(), notice, log::where_name=path, logdog::attr::tvm_guard::response=result);
        if (result.action == tvm_guard::Action::accept) {
            return std::forward<Continuation>(continuation)(std::move(parameters), stream, context);
        } else {
            return make_unexpected(error_code(Error::invalidTvm2Ticket));
        }
    }

public:
    template <class C>
    auto operator ()(C&& continuation, hana::tuple<Uid>&& parameters, const StreamPtr& stream,
            const TaskContextPtr& context) const
            -> decltype(check(std::forward<C>(continuation), std::move(parameters), {}, stream, context)) {
        const auto& uid = hana::front(parameters);
        return check(std::forward<C>(continuation), std::move(parameters), uid.value, stream, context);
    }

    template <class C>
    auto operator ()(C&& continuation, hana::tuple<>&& parameters, const StreamPtr& stream,
            const TaskContextPtr& context) const
            -> decltype(check(std::forward<C>(continuation), std::move(parameters), {}, stream, context)) {
        return check(std::forward<C>(continuation), std::move(parameters), {}, stream, context);
    }

private:
    std::shared_ptr<const TvmGuard> tvmGuard;
    std::string path;
    std::string serviceTicketHeader;
    std::string userTicketHeader;
};

template <class TvmGuard>
auto makeTvmGuarded(std::shared_ptr<const TvmGuard> tvmGuard, std::string path) {
    return TvmGuarded<std::decay_t<TvmGuard>>(std::move(tvmGuard), std::move(path));
}

} // collie::server
