#pragma once

#include "status.hpp"
#include "stream.hpp"
#include "handlers/reflection/revision_result.hpp"

#include <src/error_code.hpp>
#include <src/expected.hpp>

#include <src/services/xml.hpp>

#include <yamail/data/serialization/yajl.h>

namespace collie::server {

template<typename Serialize> class Respond {
public:
    Respond(Serialize serialize)
        : serialize(std::move(serialize)) {
    }

    template <class T>
    expected<void> operator ()(const T& value) const {
        serialize(value);
        return {};
    }

    expected<void> operator ()(logic::Revision value) const {
        return (*this)(RevisionResult {value});
    }

    expected<void> operator ()(error_code&& value) const {
        if (value) {
            return make_unexpected(std::move(value));
        } else {
            return (*this)(StatusOk {});
        }
    }

    expected<void> operator ()(const error_code& value) const {
        if (value) {
            return make_unexpected(value);
        } else {
            return (*this)(StatusOk {});
        }
    }

    expected<void> operator ()() const {
        return (*this)(StatusOk {});
    }

    template <class T>
    expected<void> operator ()(expected<T>&& value) const {
        return std::move(value).bind(*this).catch_error(*this);
    }

    template <class T>
    expected<void> operator ()(const expected<T>& value) const {
        return value.bind(*this).catch_error(*this);
    }

private:
    Serialize serialize;
};

template <class T>
[[nodiscard]]
expected<void> respond(const StreamPtr& stream, T&& value) {
    const auto serializer{[&](const auto&) {
        stream->set_code(ymod_webserver::codes::ok);
    }};

    return Respond(std::move(serializer))(std::forward<T>(value));
}

template <class T>
[[nodiscard]]
expected<void> respondWithJson(const StreamPtr& stream, T&& value) {
    const auto serializer{[&](const auto& valueToSerialize) {
        stream->set_code(ymod_webserver::codes::ok);
        stream->set_content_type("application/json");
        stream->result_body(yamail::data::serialization::toJson(valueToSerialize));
    }};

    return Respond(std::move(serializer))(std::forward<T>(value));
}

template <class T>
[[nodiscard]]
 expected<void> respondWithXml(const StreamPtr& stream, T&& value, const std::string& name,
        const std::string& attribute = {}) {
    const auto serializer{[&](const auto& valueToSerialize) {
        stream->set_code(ymod_webserver::codes::ok);
        stream->set_content_type("application/xml");

        std::string body{services::toXml(valueToSerialize, name)};
        if (!attribute.empty()) {
            body.insert(body.find(name) + name.size(), " " + attribute);
        }

        stream->result_body(body);
    }};

    return Respond(std::move(serializer))(std::forward<T>(value));
}

} // namespace collie::server
