#pragma once

#include "binary_search.hpp"
#include "error.hpp"
#include "types.hpp"

#include <boost/hana/append.hpp>
#include <boost/hana/filter.hpp>
#include <boost/hana/greater.hpp>
#include <boost/hana/ordering.hpp>
#include <boost/hana/sort.hpp>
#include <boost/hana/string.hpp>
#include <boost/hana/unique.hpp>

#include <optional>
#include <variant>

namespace boost::hana {
namespace {
using namespace router;
} // namespace

template <>
struct equal_impl<ParameterTag, ParameterTag> {
    template <class T>
    static constexpr auto apply(const Parameter<T>&, const Parameter<T>&) {
        return hana::bool_c<true>;
    }

    template <class Lhs, class Rhs>
    static constexpr auto apply(const Parameter<Lhs>&, const Parameter<Rhs>&) {
        return hana::bool_c<false>;
    }
};

template <>
struct equal_impl<HandlerTag, HandlerTag> {
    template <class T>
    static constexpr auto apply(const Handler<T>&, const Handler<T>&) {
        return hana::bool_c<true>;
    }

    template <class Lhs, class Rhs>
    static constexpr auto apply(const Handler<Lhs>&, const Handler<Rhs>&) {
        return hana::bool_c<false>;
    }
};

template <>
struct equal_impl<LocationTag, LocationTag> {
    template <class Lhs, class Rhs>
    static constexpr auto apply(const Location<Lhs>&, const Location<Rhs>&) {
        return hana::bool_c<Lhs {} == Rhs {}>;
    }
};

template <>
struct equal_impl<MiddlewareTag, MiddlewareTag> {
    template <class Lhs, class Rhs>
    static constexpr auto apply(const Middleware<Lhs>&, const Middleware<Rhs>&) {
        return hana::bool_c<true>;
    }
};

template <>
struct less_impl<LocationTag, LocationTag> {
    template <class Lhs, class Rhs>
    static constexpr auto apply(const Location<Lhs>&, const Location<Rhs>&) {
        return hana::bool_c<Lhs {} < Rhs {}>;
    }
};

template <>
struct equal_impl<MethodTag, MethodTag> {
    template <class Lhs, class Rhs>
    static constexpr auto apply(const Method<Lhs>&, const Method<Rhs>&) {
        return hana::bool_c<typename Method<Lhs>::name_type {} == typename Method<Rhs>::name_type {}>;
    }
};

template <>
struct less_impl<MethodTag, MethodTag> {
    template <class Lhs, class Rhs>
    static constexpr auto apply(const Method<Lhs>&, const Method<Rhs>&) {
        return hana::bool_c<typename Method<Lhs>::name_type {} < typename Method<Rhs>::name_type {}>;
    }
};

} // namespace boost::hana

namespace router {
namespace literals {

template <class CharT, CharT ... c>
constexpr auto operator "" _l() {
    static_assert(sizeof ... (c) > 0, "Location address should be not empty");
    return location<hana::string<c ...>>;
}

template <class CharT, CharT ... c>
constexpr auto operator "" _location() {
    static_assert(sizeof ... (c) > 0, "Location address should be not empty");
    return location<hana::string<c ...>>;
}

template <class CharT, CharT ... c>
constexpr auto operator "" _m() {
    static_assert(sizeof ... (c) > 0, "Method name should be not empty");
    return method<hana::string<c ...>>;
}

template <class CharT, CharT ... c>
constexpr auto operator "" _method() {
    static_assert(sizeof ... (c) > 0, "Method name should be not empty");
    return method<hana::string<c ...>>;
}

} // namespace literals

template <class Lhs, class Rhs>
constexpr bool operator ==(const Tree<Lhs>& lhs, const Tree<Rhs>& rhs) {
    return lhs.value == rhs.value;
}

template <class Lhs, class Rhs>
constexpr bool operator !=(const Tree<Lhs>& lhs, const Tree<Rhs>& rhs) {
    return !(lhs == rhs);
}

template <class Lhs, class Rhs, class = std::enable_if_t<is_tree_node_v<Lhs> && is_tree_node_v<Rhs>, void*>>
constexpr auto operator /(const Lhs& lhs, const Rhs& rhs) {
    return tree(hana::make_tuple(hana::make_pair(lhs, rhs)));
}

template <class Lhs, class Rhs,
          class = std::enable_if_t<is_tree_node_v<Lhs> && is_part_of_tree_v<Rhs>, void*>>
constexpr auto operator /(const Tree<Lhs>& lhs, const Rhs& rhs) {
    return lhs.value / rhs;
}

template <class Lhs, class Rhs, class = std::enable_if_t<is_tree_node_v<Lhs>, void*>>
constexpr auto operator /(const Lhs& lhs, const Tree<Rhs>& rhs) {
    return tree(hana::make_tuple(hana::make_pair(lhs, rhs.value)));
}

template <class Lhs1, class Lhs2, class Rhs, class = std::enable_if_t<is_part_of_tree_v<Rhs>, void*>>
constexpr auto operator /(const hana::pair<Lhs1, Lhs2>& lhs, const Rhs& rhs) {
    return tree(hana::make_tuple(hana::make_pair(hana::first(lhs), (tree(hana::second(lhs)) / rhs).value)));
}

template <class Lhs, class Rhs, class = std::enable_if_t<is_part_of_tree_v<Rhs>, void*>>
constexpr auto operator /(const Tree<hana::tuple<Lhs>>& lhs, const Rhs& rhs) {
    return hana::front(lhs.value) / rhs;
}

template <class ... Lhs, class ... Rhs>
constexpr auto operator |(const Tree<hana::tuple<Lhs ...>>& lhs, const Tree<hana::tuple<Rhs ...>>& rhs) {
    static_assert(
        std::is_same_v<
            decltype(hana::sort.by(
                hana::ordering(hana::first),
                hana::filter(hana::concat(lhs.value, rhs.value), is_location)
            )),
            decltype(hana::unique(hana::sort.by(
                hana::ordering(hana::first),
                hana::filter(hana::concat(lhs.value, rhs.value), is_location)
            )))
        >,
        "locations should be without duplicates"
    );
    static_assert(
        std::is_same_v<
            decltype(hana::sort.by(
                hana::ordering(hana::first),
                hana::filter(hana::concat(lhs.value, rhs.value), is_method)
            )),
            decltype(hana::unique(hana::sort.by(
                hana::ordering(hana::first),
                hana::filter(hana::concat(lhs.value, rhs.value), is_method)
            )))
        >,
        "methods should be without duplicates"
    );
    static_assert(
        !(
            !std::is_same_v<
                decltype(hana::filter(hana::concat(lhs.value, rhs.value), is_method)),
                hana::tuple<>
            >
            && !std::is_same_v<
                decltype(hana::filter(hana::concat(lhs.value, rhs.value), is_middleware)),
                hana::tuple<>
            >
        ),
        "parameter and middleware should not be in one tuple"
    );
    return tree(
        hana::concat(
            hana::sort.by(
                hana::ordering(hana::first),
                hana::concat(hana::filter(lhs.value, is_location), hana::filter(rhs.value, is_location))
            ),
            hana::concat(
                hana::sort.by(
                    hana::ordering(hana::first),
                    hana::concat(hana::filter(lhs.value, is_method), hana::filter(rhs.value, is_method))
                ),
                hana::concat(
                    hana::filter(lhs.value, is_handler),
                    hana::concat(
                        hana::filter(rhs.value, is_handler),
                        hana::concat(
                            hana::filter(lhs.value, is_parameter),
                            hana::concat(
                                hana::filter(rhs.value, is_parameter),
                                hana::concat(
                                    hana::filter(lhs.value, is_middleware),
                                    hana::filter(rhs.value, is_middleware)
                                )
                            )
                        )
                    )
                )
            )
        )
    );
}

struct Less {
    template <class MethodName, class First, class Second>
    bool operator ()(MethodName method, const hana::pair<Method<First>, Second>& value) const {
        return method < get_name(hana::first(value));
    }

    template <auto ... first, class Second>
    bool operator ()(const std::string& method,
            const hana::pair<Method<hana::string<first ...>>, Second>& value) const {
        return (*this)(std::string_view(method), value);
    }

    template <auto ... first, class Second>
    bool operator ()(const std::string_view& method,
            const hana::pair<Method<hana::string<first ...>>, Second>& value) const {
        const auto name = hana::to<const char*>(get_name(hana::first(value)));
        return std::lexicographical_compare(
            std::begin(method), std::end(method),
            name, name + hana::value(hana::length(get_name(hana::first(value))))
        );
    }

    template <char ... c, auto ... first, class Second>
    bool operator ()(hana::string<c ...> method,
            const hana::pair<Method<hana::string<first ...>>, Second>& value) const {
        const auto name = hana::to<const char*>(get_name(hana::first(value)));
        return std::lexicographical_compare(
            hana::to<const char*>(method), hana::to<const char*>(method) + hana::value(hana::length(method)),
            name, name + hana::value(hana::length(get_name(hana::first(value))))
        );
    }

    template <class Lhs, class First, class Second>
    bool operator ()(const Lhs& name, const hana::pair<Location<First>, Second>& value) const {
        return name < get_address(hana::first(value));
    }

    template <class First, class Second>
    bool operator ()(const std::string& name, const hana::pair<Location<First>, Second>& value) const {
        return name < hana::to<const char*>(get_address(hana::first(value)));
    }
};

struct Equal {
    template <class MethodName, class First, class Second>
    bool operator ()(MethodName method, const hana::pair<Method<First>, Second>& value) const {
        return method == get_name(hana::first(value));
    }

    template <auto ... first, class Second>
    bool operator ()(const std::string& method,
            const hana::pair<Method<hana::string<first ...>>, Second>& value) const {
        return (*this)(std::string_view(method), value);
    }

    template <auto ... first, class Second>
    bool operator ()(const std::string_view& method,
            const hana::pair<Method<hana::string<first ...>>, Second>& value) const {
        const auto name = hana::to<const char*>(get_name(hana::first(value)));
        return std::equal(
            std::begin(method), std::end(method),
            name, name + hana::value(hana::length(get_name(hana::first(value))))
        );
    }

    template <char ... c, auto ... first, class Second>
    bool operator ()(hana::string<c ...> method,
            const hana::pair<Method<hana::string<first ...>>, Second>& value) const {
        const auto name = hana::to<const char*>(get_name(hana::first(value)));
        return std::equal(
            hana::to<const char*>(method), hana::to<const char*>(method) + hana::value(hana::length(method)),
            name, name + hana::value(hana::length(get_name(hana::first(value))))
        );
    }

    template <class Lhs, class First, class Second>
    bool operator ()(const Lhs& name, const hana::pair<Location<First>, Second>& value) const {
        return name == get_address(hana::first(value));
    }

    template <class First, class Second>
    bool operator ()(const std::string& name, const hana::pair<Location<First>, Second>& value) const {
        return name == hana::to<const char*>(get_address(hana::first(value)));
    }
};

template <class F>
constexpr auto return_monostate_on_void(F&& f) {
    return [f = std::forward<F>(f)] (const auto& next) {
        if constexpr (std::is_same_v<decltype(f(next)), void>) {
            f(next);
            return std::monostate {};
        } else {
            return f(next);
        }
    };
}

template <class T>
using ResultType = std::conditional_t<std::is_same_v<T, void>, std::monostate, T>;

template <class T>
using CallResult = std::variant<ResultType<T>, Error>;

template <class PathIt, class MethodName>
struct Context {
    const PathIt path_it;
    const PathIt path_end;
    const MethodName& method;

    friend constexpr auto advanced(const Context& context) {
        return Context {std::next(context.path_it), context.path_end, context.method};
    }

    friend constexpr auto at_path_end(const Context& context) {
        return context.path_it == context.path_end;
    }
};

template <class Result, class Search, class Context, class Parameters, class Node, class ... Args>
constexpr CallResult<Result> route_for_tuple(const Context& context,
        const Node& node, Parameters&& parameters, Args&& ... args);

template <class Result, class Search, class Context, class Parameters, class ... Args>
constexpr CallResult<Result> route_for_parameter(const Context&, const hana::tuple<>&, Parameters&&,
        Args&& ...) noexcept {
    return Error::location_not_found;
}

template <class T>
constexpr const T* make_location(const T& value) {
    return &value;
}

template <class L, class P>
constexpr auto make_location(const std::variant<L, P>& value) {
    return std::get_if<0>(&value);
}
template <class Parameter, class Value>
constexpr auto make_parameter(const Value& value) {
    return std::make_optional(Parameter::make(value));
}

template <class Parameter, class L, class P>
constexpr auto make_parameter(const std::variant<L, P>& value)
        -> std::optional<decltype(Parameter::make(std::get<1>(value)))> {
    if (const auto v = std::get_if<1>(&value)) {
        return std::make_optional(Parameter::make(*v));
    } else {
        return {};
    }
}

template <class Result, class Parameter, class Continuation>
struct ParameterVisitor {
    Continuation continuation;

    CallResult<Result> operator ()(ResultType<Result>&& result) const {
        return std::move(result);
    }

    CallResult<Result> operator ()(Parameter&& value) const {
        return continuation(std::move(value));
    }
};

template <class Result, class Parameter, class Continuation>
constexpr auto make_parameter_visitor(Continuation&& continuation) {
    return ParameterVisitor<Result, Parameter, std::decay_t<Continuation>> {std::forward<Continuation>(continuation)};
}

template <class Result, class Search, class Context, class Parameters, class T, class Node, class ... Args>
constexpr CallResult<Result> route_for_constructed_parameter(const Context& context,
        const hana::tuple<hana::pair<Parameter<T>, Node>>& node, T&& value,
        Parameters&& parameters, Args&& ... args) {
    return route_for_tuple<Result, Search>(
        advanced(context),
        hana::second(hana::front(node)),
        hana::append(std::forward<Parameters>(parameters), std::move(value)),
        std::forward<Args>(args) ...
    );
}

template <class Result, class Search, class Context, class Parameters, class T, class Node, class ... V, class ... Args>
constexpr CallResult<Result> route_for_constructed_parameter(const Context& context,
        const hana::tuple<hana::pair<Parameter<T>, Node>>& node, std::variant<V ...>&& value,
        Parameters&& parameters, Args&& ... args) {
    return std::visit(make_parameter_visitor<Result, T>(
        [&] (auto&& value) {
            return route_for_tuple<Result, Search>(
                advanced(context),
                hana::second(hana::front(node)),
                hana::append(std::forward<Parameters>(parameters), std::move(value)),
                std::forward<Args>(args) ...
            );
        }
    ), std::move(value));
}

template <class Result, class Search, class Context, class Parameters, class T, class Node, class ... Args>
constexpr CallResult<Result> route_for_parameter(const Context& context,
        const hana::tuple<hana::pair<Parameter<T>, Node>>& node,
        Parameters&& parameters, Args&& ... args) {
    if (auto value = make_parameter<T>(*context.path_it)) {
        return route_for_constructed_parameter<Result, Search>(
            context,
            node,
            *std::move(value),
            std::forward<Parameters>(parameters),
            std::forward<Args>(args) ...
        );
    } else {
        return Error::location_not_found;
    }
}

template <class Result, class Search, class Context, class Parameters, class ... Args>
constexpr std::optional<CallResult<Result>> route_for_location(const Context&,
        const hana::tuple<>&, Parameters&&, Args&& ...) {
    return {};
}

template <class Result, class Search, class Context, class Parameters, class ... T, class ... Node, class ... Args>
constexpr std::optional<CallResult<Result>> route_for_location(const Context& context,
        const hana::tuple<hana::pair<Location<T>, Node> ...>& node,
        Parameters&& parameters, Args&& ... args) {
    if (const auto value = make_location(*context.path_it)) {
        return Search::template apply<CallResult<Result>>(
            node,
            *value,
            Less {},
            Equal {},
            return_monostate_on_void([&] (const auto& next) {
                return route_for_tuple<Result, Search>(
                    advanced(context),
                    hana::second(next),
                    std::forward<Parameters>(parameters),
                    std::forward<Args>(args) ...
                );
            })
        );
    } else {
        return {};
    }
}

template <class Result, class Search, class Context, class Parameters, class ... Args>
constexpr CallResult<Result> route_for_method(const Context&,
        const hana::tuple<>&, Parameters&&, Args&& ...) noexcept {
    return Error::method_not_found;
}

template <class Result, class Search, class Context, class Parameters, class ... MethodName, class ... Node, class ... Args>
constexpr CallResult<Result> route_for_method(const Context& context,
        const hana::tuple<hana::pair<Method<MethodName>, Node> ...>& node,
        Parameters&& parameters, Args&& ... args) {
    auto result = Search::template apply<CallResult<Result>>(
        node,
        context.method,
        Less {},
        Equal {},
        return_monostate_on_void([&] (const auto& next) {
            return hana::unpack(std::forward<Parameters>(parameters), [&] (auto&& ... params) {
                return hana::second(next).impl(std::move(params) ..., std::forward<Args>(args) ...);
            });
        })
    );
    return result ? std::move(result).value() : Error::method_not_found;
}

template <class Result, class Search, class Context, class Parameters, class T, class Node, class ... Args>
constexpr CallResult<Result> route_for_middleware(const Context& context,
        const hana::tuple<hana::pair<Middleware<T>, Node>>& node,
        Parameters&& parameters, Args&& ... args) {
    return return_monostate_on_void([&] (const auto& next) {
        return hana::first(next).impl(
            [&] (auto&& new_parameters, auto&& ... new_args) {
                return route_for_tuple<Result, Search>(
                    context,
                    hana::second(next),
                    std::move(new_parameters),
                    std::move(new_args) ...
                );
            },
            std::forward<Parameters>(parameters),
            std::forward<Args>(args) ...
        );
    })(hana::front(node));
}

template <class Result, class Search, class Context, class Parameters, class ... Key, class ... Node, class ... Args>
constexpr CallResult<Result> route_for_tuple(const Context& context,
        const hana::tuple<hana::pair<Key, Node> ...>& node,
        Parameters&& parameters, Args&& ... args) {
    if (!at_path_end(context)) {
        if (auto result = route_for_location<Result, Search>(
                context,
                hana::filter(node, is_location),
                std::forward<Parameters>(parameters),
                std::forward<Args>(args) ...
        )) {
            return std::move(result).value();
        }
        if constexpr (!std::is_same_v<decltype(hana::filter(node, is_parameter)), hana::tuple<>>) {
            return route_for_parameter<Result, Search>(
                context,
                hana::filter(node, is_parameter),
                std::forward<Parameters>(parameters),
                std::forward<Args>(args) ...
            );
        } else if constexpr (!std::is_same_v<decltype(hana::filter(node, is_middleware)), hana::tuple<>>) {
            return route_for_middleware<Result, Search>(
                context,
                hana::filter(node, is_middleware),
                std::forward<Parameters>(parameters),
                std::forward<Args>(args) ...
            );
        } else {
            return Error::location_not_found;
        }
    }
    if constexpr (!std::is_same_v<decltype(hana::filter(node, is_method)), hana::tuple<>>) {
        return route_for_method<Result, Search>(
            context,
            hana::filter(node, is_method),
            std::forward<Parameters>(parameters),
            std::forward<Args>(args) ...
        );
    } else if constexpr (!std::is_same_v<decltype(hana::filter(node, is_middleware)), hana::tuple<>>) {
        return route_for_middleware<Result, Search>(
            context,
            hana::filter(node, is_middleware),
            std::forward<Parameters>(parameters),
            std::forward<Args>(args) ...
        );
    } else {
        return Error::method_not_found;
    }
}

template <class Result, class Search, class T, class Path, class MethodName, class ... Args>
constexpr CallResult<Result> route_with_search(const Tree<T>& tree, const Path& path,
        const MethodName& method, Args&& ... args) {
    using ContextType = Context<typename Path::const_iterator, std::decay_t<MethodName>>;
    return route_for_tuple<Result, Search>(
        ContextType {std::cbegin(path), std::cend(path), method},
        tree.value,
        hana::tuple<>(),
        std::forward<Args>(args) ...
    );
}

template <class Result, class T, class Path, class MethodName, class ... Args>
constexpr CallResult<Result> route(const Tree<T>& tree, const Path& path,
        const MethodName& method, Args&& ... args) {
    return route_with_search<Result, BinarySearch>(tree, path, method, std::forward<Args>(args) ...);
}

} // namespace router
