#pragma once

#include <boost/hana/core/tag_of.hpp>
#include <boost/hana/tuple.hpp>

#include <type_traits>
#include <utility>

namespace router {

namespace hana = boost::hana;

struct TreeTag {};

template <class T>
struct Tree {
    using hana_tag = TreeTag;
    T value;
};

template <class T>
constexpr auto tree(T&& value) {
    return Tree<std::decay_t<T>> {std::forward<T>(value)};
}

template <class T>
struct IsTree : std::integral_constant<bool, std::is_same<hana::tag_of_t<T>, TreeTag>::value> {};

template <class T>
constexpr const bool is_tree_v = IsTree<T>::value;

struct ParameterTag {};

template <class TypeT>
struct Parameter {
    using Type = TypeT;
    using hana_tag = ParameterTag;
};

template <class T>
constexpr auto parameter = Parameter<std::decay_t<T>> {};

template <class T>
struct IsParameter : std::integral_constant<bool, std::is_same<hana::tag_of_t<T>, ParameterTag>::value> {};

template <class T>
constexpr const bool is_parameter_v = IsParameter<T>::value;

constexpr auto is_parameter = [] (const auto& v) { return IsParameter<std::decay_t<decltype(hana::first(unref(v)))>> {}; };

struct HandlerTag {};

template <class T>
struct Handler {
    using hana_tag = HandlerTag;
    T impl;
};

template <class T>
constexpr auto handler(T&& value) {
    return Handler<std::decay_t<T>> {std::forward<T>(value)};
};

template <class T>
struct IsHandler : std::integral_constant<bool, std::is_same<hana::tag_of_t<T>, HandlerTag>::value> {};

template <class T>
constexpr const bool is_handler_v = IsHandler<T>::value;

constexpr auto is_handler = [] (const auto& v) { return IsHandler<std::decay_t<decltype(hana::first(unref(v)))>> {}; };

struct LocationTag {};

template <class Address>
struct Location {
    using address_type = Address;
    using hana_tag = LocationTag;
};

template <class T>
constexpr auto location = Location<std::decay_t<T>> {};

template <class T>
struct IsLocation : std::integral_constant<bool, std::is_same<hana::tag_of_t<T>, LocationTag>::value> {};

template <class T>
constexpr const bool is_location_v = IsLocation<T>::value;

constexpr auto is_location = [] (const auto& v) { return IsLocation<std::decay_t<decltype(hana::first(unref(v)))>> {}; };

template <class T>
constexpr auto get_address(Location<T>) noexcept(noexcept(typename Location<T>::address_type {})) {
    return typename Location<T>::address_type {};
}

template <class T>
constexpr auto get_address(std::reference_wrapper<Location<T>>) noexcept(noexcept(typename Location<T>::address_type {})) {
    return typename Location<T>::address_type {};
}

template <class T>
constexpr auto get_address(std::reference_wrapper<const Location<T>>) noexcept(noexcept(typename Location<T>::address_type {})) {
    return typename Location<T>::address_type {};
}

struct MethodTag {};

template <class Name>
struct Method {
    using name_type = Name;
    using hana_tag = MethodTag;

    template <class T>
    constexpr auto operator ()(T&& v) const {
        return tree(hana::make_tuple(hana::make_pair(*this, handler(std::forward<T>(v)))));
    }
};

template <class T>
constexpr auto method = Method<std::decay_t<T>> {};

template <class T>
constexpr auto make_method(T&&) noexcept {
    return method<T>;
}

template <class T>
struct IsMethod : std::integral_constant<bool, std::is_same<hana::tag_of_t<T>, MethodTag>::value> {};

template <class T>
constexpr const bool is_method_v = IsMethod<T>::value;

constexpr auto is_method = [] (const auto& v) { return IsMethod<std::decay_t<decltype(hana::first(unref(v)))>> {}; };

template <class T>
constexpr auto get_name(Method<T>) noexcept(noexcept(typename Method<T>::name_type {})) {
    return typename Method<T>::name_type {};
}

template <class T>
constexpr auto get_name(std::reference_wrapper<Method<T>>) noexcept(noexcept(typename Method<T>::name_type {})) {
    return typename Method<T>::name_type {};
}

template <class T>
constexpr auto get_name(std::reference_wrapper<const Method<T>>) noexcept(noexcept(typename Method<T>::name_type {})) {
    return typename Method<T>::name_type {};
}

struct MiddlewareTag {};

template <class T>
struct Middleware {
    using hana_tag = MiddlewareTag;

    T impl;
};

template <class T>
constexpr auto middleware(T&& impl) noexcept(noexcept(Middleware<std::decay_t<T>> {std::forward<T>(impl)})) {
    return Middleware<std::decay_t<T>> {std::forward<T>(impl)};
}

template <class T>
struct IsMiddleware : std::integral_constant<bool, std::is_same<hana::tag_of_t<T>, MiddlewareTag>::value> {};

template <class T>
constexpr const bool is_middleware_v = IsMiddleware<T>::value;

constexpr auto is_middleware = [] (const auto& v) { return IsMiddleware<std::decay_t<decltype(hana::first(unref(v)))>> {}; };

template <class T>
constexpr const bool is_tree_node_v = is_handler_v<T>
    || is_location_v<T>
    || is_parameter_v<T>
    || is_method_v<T>
    || is_middleware_v<T>;

template <class T>
constexpr const bool is_part_of_tree_v = is_tree_node_v<T> || is_tree_v<T>;

template <class T>
constexpr decltype(auto) unref(const T& value) noexcept {
    return value;
}

template <class T>
constexpr decltype(auto) unref(const std::reference_wrapper<T>& value) noexcept {
    return value.get();
}

} // namespace router
