#pragma once

#include "config.hpp"
#include "status.hpp"
#include "utils.hpp"
#include "router/router.hpp"
#include "router/error_category.hpp"

#include <src/expected.hpp>
#include <src/log.hpp>
#include <src/task_context.hpp>

#include <sharpei_client/errors.h>

#include <ymod_webserver/server.h>
#include <ymod_webserver/response.h>

#include <yplatform/reactor.h>
#include <yplatform/find.h>

#include <yamail/data/serialization/yajl.h>

#include <boost/asio/spawn.hpp>

namespace collie::server {

ymod_webserver::codes::code getYmodWebserverCode(const router::Error value);
ymod_webserver::codes::code getYmodWebserverCode(const Error value);
ymod_webserver::codes::code getYmodWebserverCode(const sharpei::client::Errors value);
ymod_webserver::codes::code getYmodWebserverCode(const mail_errors::error_code& value);
Error getError(const mail_errors::error_code& value);

template <class RequestLogger>
struct ResponseError {
    const StreamPtr& stream;
    const RequestLogger& requestLogger;

    void operator ()(router::Error result) const {
        using yamail::data::serialization::toJson;
        const std::error_code Error(result);
        LOGDOG_(requestLogger, error,
            log::message="error while dispatching request",
            log::error_code=Error
        );
        const auto body = toJson(StatusError {Error::routeError, Error.message()}).str();
        stream->set_code(getYmodWebserverCode(result));
        stream->set_content_type("application/json");
        stream->result_body(body);
    }

    void operator ()(mail_errors::error_code error) const {
        using yamail::data::serialization::toJson;

        if (!error) {
            return;
        }

        LOGDOG_(requestLogger, error,
            log::message="failed to handle request",
            logdog::attr::error_code=error
        );
        const auto body = toJson(StatusError {getError(error), error.message()}).str();
        stream->set_code(getYmodWebserverCode(error));
        stream->set_content_type("application/json");
        stream->result_body(body);
    }

    void operator ()(Error error, std::string message) const {
        using yamail::data::serialization::toJson;

        const auto body = toJson(StatusError {error, std::move(message)}).str();
        stream->set_code(getYmodWebserverCode(error));
        stream->set_content_type("application/json");
        stream->result_body(body);
    }

    void operator ()(expected<void>&& result) const {
        std::move(result).catch_error(*this);
    }
};

template <class Api, class Logger, class GetIo>
class Handler final : public ymod_webserver::handler {
public:
    Handler(const Api& api, std::size_t coroutineStackSize, Logger logger, GetIo getIo)
        : api(api),
          coroutineStackSize(coroutineStackSize),
          logger(logger),
          getIo(getIo) {}

    void execute(ymod_webserver::request_ptr /*request*/, StreamPtr stream) override {
        using yamail::data::serialization::toJson;
        const auto io = getIo();
        const auto requestId = getRequestId(*stream);
        boost::asio::spawn(*io, [=] (boost::asio::yield_context yield) {
            const std::string requestIdStr(requestId);
            const auto& streamContext = stream->ctx();
            auto requestLogger = logdog::bind(logger,
                log::request_id=requestIdStr,
                log::uniq_id=streamContext->uniq_id()
            );
            const auto logException = [&] (const auto& e) { collie::logException(requestLogger, e); };
            const ResponseError<decltype(requestLogger)> responseError {stream, requestLogger};
            try {
                const auto context = boost::make_shared<TaskContext>(
                    streamContext->uniq_id(),
                    requestIdStr,
                    "user_ip",
                    "client_type",
                    yield
                );
                const auto& request = stream->request();
                std::visit(
                    responseError,
                    router::route<expected<void>>(
                        api,
                        request->url.path,
                        request->method,
                        stream,
                        context
                    )
                );
            } catch (const boost::coroutines::detail::forced_unwind&) {
                throw;
            } catch (const std::exception& e) {
                logException(e);
                responseError(Error::unclassifiedError, e.what());
            } catch (...) {
                LOGDOG_(requestLogger, error, log::message="unknown error while dispatching request");
                responseError(Error::unclassifiedError, "unknown error");
            }
        }, boost::coroutines::attributes(coroutineStackSize));
    }

private:
    Api api;
    std::size_t coroutineStackSize;
    Logger logger;
    GetIo getIo;
};

template <class Api, class Logger, class GetIo>
inline constexpr auto makeHandler(Api&& resourceTree, std::size_t coroutineStackSize,
        Logger&& logger, GetIo&& getIo) {
    return boost::make_shared<Handler<std::decay_t<Api>, std::decay_t<Logger>, std::decay_t<GetIo>>>(
        std::forward<Api>(resourceTree),
        coroutineStackSize,
        std::forward<Logger>(logger),
        std::forward<GetIo>(getIo)
    );
}

template <class Api>
inline void init(const Config& config, Api&& resourceTree) {
    const auto server = yplatform::find<ymod_webserver::server, std::shared_ptr>(config.webServerModule);
    const auto getIo = [] { return yplatform::global_net_reactor->io(); };
    server->subscribe("",
        makeHandler(
            std::forward<Api>(resourceTree),
            config.coroutineStackSize,
            getLogger(),
            getIo
        )
    );
}

} // namespace collie::server
