#pragma once

#include "connection_provider.hpp"
#include "error.hpp"
#include "retry.hpp"
#include "utils.hpp"

#include <src/expected.hpp>
#include <src/log.hpp>

#include <ozo/transaction.h>

namespace collie::services::db {

namespace adl {

template <typename T>
struct BeginImpl {
    template <typename ...Ts>
    static auto apply(Ts&& ...v) { return ozo::begin(std::forward<Ts>(v)...);}
};

template <class T, class ...Ts>
auto begin(T&& v, Ts&& ...vs) {
    return BeginImpl<std::decay_t<T>>::apply(std::forward<T>(v), std::forward<Ts>(vs)...);
}

} // namespace adl

template <class T>
struct Begin {
    static auto apply(T provider) {
        using adl::begin;
        using ozo::get_error_context;
        using ozo::error_message;

        static_assert(ConnectionProvider<T>);

        const auto requestTimeout = provider.requestTimeout();
        const auto context = provider.context();

        LOGDOG_(context->logger(), notice, log::query="BEGIN");

        ozo::error_code ec;
        auto conn = begin(provider, requestTimeout, context->yield()[ec]);

        if (!ec) {
            return expected(std::move(conn));
        }

        LOGDOG_(context->logger(), error,
            log::error_code=ec,
            log::query="BEGIN",
            log::message=(conn ? get_error_context(conn) : std::string()),
            log::pq_message=(conn ? error_message(conn) : std::string_view())
        );

        if (!userNotFound(ec)) {
            ec = Error::databaseError;
        }

        return make_expected_from_error<decltype(conn)>(error_code{std::move(ec)});
    }
};

template <class T, class = std::enable_if_t<ConnectionProvider<T> || Retriable<T>>>
auto begin(T&& provider) {
    return Begin<std::decay_t<T>>::apply(std::forward<T>(provider));
}

template <class T>
struct Begin<Retry<T>> {
    static auto apply(Retry<T> provider) {
        static_assert(ConnectionProvider<T>);

        const auto maxRetriesNumber = db::unwrap(provider).maxRetriesNumber();
        std::size_t tryNumber = 0;

        while (true) {
            const auto result = begin(db::unwrap(provider));

            if (result) {
                return result;
            }

            if (!isRetriable(result.error()) || ++tryNumber > maxRetriesNumber) {
                return result;
            }
        }
    }
};

} // namespace collie::services::db
