#pragma once

#include "connection_provider.hpp"
#include "error.hpp"
#include "retry.hpp"

#include <src/log.hpp>

#include <ozo/transaction.h>

namespace collie::services::db {

template <class T>
struct Commit {
    static_assert(ConnectionProvider<T>);

    template <class Transaction>
    static auto apply(T provider, Transaction&& transaction) {
        const auto requestTimeout = provider.requestTimeout();
        const auto context = provider.context();

        LOGDOG_(context->logger(), notice, log::query="COMMIT");

        ozo::error_code ec;
        auto conn = ozo::commit(std::forward<Transaction>(transaction), requestTimeout, context->yield()[ec]);

        if (ec) {
            LOGDOG_(context->logger(), error,
                log::error_code=ec,
                log::query="COMMIT",
                log::message=ozo::get_error_context(conn)
            );
            return make_expected_from_error<decltype(conn)>(error_code(std::move(ec)));
        }

        return expected(std::move(conn));
    }
};

template <class T, class Transaction>
auto commit(T&& provider, Transaction&& transaction) {
    return Commit<std::decay_t<T>>::apply(std::forward<T>(provider), std::forward<Transaction>(transaction));
}

} // namespace collie::services::db
