#pragma once

#include <ozo/connection.h>

namespace collie::services::db {

template <typename T, typename = std::void_t<>>
struct error_context_type {
    using type = typename T::error_context;
};

template <typename T>
struct error_context_type<T, std::void_t<typename T::error_context_type>> {
    using type = typename T::error_context_type;
};
template <typename Context, typename Connection>
struct ConnectionAdaptor : Context {
    using target_type = std::decay_t<decltype(ozo::unwrap_connection(std::declval<Connection>()))>;
    using native_handle_type = typename target_type::native_handle_type;
    using oid_map_type = typename target_type::oid_map_type;
    using error_context_type = typename error_context_type<target_type>::type;
    using executor_type = typename target_type::executor_type;

    Connection target_;

    ConnectionAdaptor(Context ctx, Connection target)
    : Context(std::move(ctx)), target_(std::move(target)) {}

    ConnectionAdaptor() = default;

    native_handle_type native_handle() const noexcept { return target().native_handle(); }

    const oid_map_type& oid_map() const noexcept { return target().oid_map();}

    decltype(auto) statistics() const noexcept { return target().statistics();}

    const error_context_type& get_error_context() const noexcept { return target().get_error_context(); }
    void set_error_context(error_context_type v = {}) { target().set_error_context(std::move(v)); }

    executor_type get_executor() const noexcept { return target().get_executor(); }

    template <typename WaitHandler>
    void async_wait_write(WaitHandler&& h) { target().async_wait_write(std::forward<WaitHandler>(h));}

    template <typename WaitHandler>
    void async_wait_read(WaitHandler&& h) { target().async_wait_read(std::forward<WaitHandler>(h));}

    ozo::error_code close() noexcept { return target().close();}

    void cancel() noexcept { target().cancel();}

    bool is_bad() const noexcept { return target().is_bad();}

    bool is_open() const noexcept { return target().is_open();}

    operator bool() const noexcept { return static_cast<bool>(target());}

    bool is_null() const noexcept { return ozo::is_null_recursive(target_);}

    target_type& target() noexcept { return ozo::unwrap_connection(target_); }
    const target_type& target() const noexcept { return ozo::unwrap_connection(target_); }
};

} // namespace collie::services::db

namespace ozo {

template <typename ...Ts>
struct is_nullable<collie::services::db::ConnectionAdaptor<Ts...>> : std::true_type {};

template <typename ...Ts>
struct is_null_impl<collie::services::db::ConnectionAdaptor<Ts...>> {
    static constexpr bool apply(const collie::services::db::ConnectionAdaptor<Ts...>& v) noexcept {
        return v.is_null();
    }
};

template <typename ...Ts>
struct is_connection<collie::services::db::ConnectionAdaptor<Ts...>> : std::true_type {};

} // namespace ozo
