#pragma once

#include <ozo/connection.h>
#include <ozo/transaction.h>

namespace collie::services::db {

template <class T, class = std::void_t<>>
struct IsConnectionProvider : std::false_type {};

template <class T>
struct IsConnectionProvider<T, std::void_t<
    decltype(std::declval<T&>().context()),
    decltype(std::declval<T&>().maxRetriesNumber()),
    decltype(std::declval<T&>().requestTimeout()),
    decltype(std::declval<T&>().queryRepository())
>> : ozo::is_connection_provider<T> {};

template <class T>
const auto ConnectionProvider = IsConnectionProvider<std::decay_t<T>>::value;

template <class T, class = std::enable_if_t<ConnectionProvider<T>>>
decltype(auto) unwrap(T&& value) {
    return std::forward<T>(value);
}

template <typename T, typename = std::void_t<>>
struct has_lowest_layer : std::false_type {};

template <typename T>
struct has_lowest_layer<T, std::void_t<decltype(std::declval<T>().lowest_layer())>> : std::true_type {};

template <class ...Ts>
decltype(auto) unwrap(const ozo::impl::transaction<Ts...>& value) {
    if constexpr (has_lowest_layer<const ozo::impl::transaction<Ts...>&>::value) {
        return value.lowest_layer();
    } else {
        return ozo::unwrap_connection(value);
    }
}

template <class ...Ts>
decltype(auto) unwrap(ozo::impl::transaction<Ts...>& value) {
    if constexpr (has_lowest_layer<ozo::impl::transaction<Ts...>&>::value) {
        return value.lowest_layer();
    } else {
        return ozo::unwrap_connection(value);
    }
}

template <class T>
struct Retry {
    static_assert(ConnectionProvider<T>);

    using connection_type = ozo::connection_type<T>;

    T impl;
};

template <class T>
auto retry(T&& impl) {
    return Retry<std::decay_t<T>> {std::forward<T>(impl)};
}

template <class T>
struct IsRetriable : std::false_type {};

template <class T>
struct IsRetriable<Retry<T>> : std::true_type {};

template <class T>
const auto Retriable = IsRetriable<std::decay_t<T>>::value;

template <class T, class = std::enable_if_t<ConnectionProvider<T>>>
decltype(auto) unwrap(const Retry<T>& value) {
    return value.impl;
}

template <class T, class = std::enable_if_t<ConnectionProvider<T>>>
decltype(auto) unwrap(Retry<T>& value) {
    return value.impl;
}

} // namespace collie::services::db
