#pragma once

#include "query.hpp"

#include <src/services/db/request.hpp>

#include <ozo/shortcuts.h>

namespace collie::services::db {

#include <boost/asio/yield.hpp>

namespace query = collie::services::db::contacts::query;

template <typename Handler, typename QueryRepository>
class CheckIsUserExistsOperation : public boost::asio::coroutine {
private:
    using Result = std::vector<query::IsUserExists::result_type>;

    QueryRepository queryRepository;
    std::int64_t uid;
    ConstUserType userType;
    ozo::time_traits::duration timeout;
    Handler handler;
    std::shared_ptr<Result> result;

public:
    CheckIsUserExistsOperation(QueryRepository queryRepository,
        std::int64_t uid,
        ConstUserType userType,
        ozo::time_traits::duration timeout,
        Handler handler)
      : queryRepository(queryRepository),
        uid(uid),
        userType(userType),
        timeout(timeout),
        handler(handler) {}

    template <typename Conn>
    void operator () (ozo::error_code ec, Conn&& conn) {
        if (ec) {
            return handler(ec, std::forward<Conn>(conn));
        }

        reenter(*this) {
            result = std::make_shared<Result>();

            using adl::request;
            yield request(
                std::forward<Conn>(conn),
                queryRepository.template make_query<query::IsUserExists>({uid, userType}),
                timeout,
                ozo::into(*result),
                *this
            );

            if (result->size() == 1 && std::get<0>(result->front())) {
                return handler(ec, std::forward<Conn>(conn));
            } else {
                return handler(ozo::error_code{collie::logic::Error::userNotFound}, std::forward<Conn>(conn));
            }
        }
    }

    using executor_type = decltype(boost::asio::get_associated_executor(handler));

    executor_type get_executor() const noexcept {
        return boost::asio::get_associated_executor(handler);
    }

    using allocator_type = decltype(boost::asio::get_associated_allocator(handler));

    allocator_type get_allocator() const noexcept {
        return boost::asio::get_associated_allocator(handler);
    }
};

#include <boost/asio/unyield.hpp>

} // namespace collie::services::db
