#pragma once

#include "oid_map.hpp"

#include <src/services/db/connection_provider.hpp>
#include <src/services/db/connection_adapter.hpp>
#include <src/services/db/user_type.hpp>
#include <src/task_context.hpp>

#include <ozo/connection_pool.h>

namespace collie::services::db::contacts {

using Connection = ozo::connection_type<ozo::connection_pool<ozo::connection_info<OidMap>>>;

struct ConnectionContext {
    std::int64_t uid_;
    ConstUserType userType_;
    ozo::time_traits::duration timeout_;
    std::size_t maxRetriesNumber_;
    QueryRepository queryRepository_;
    TaskContextPtr context_;

    std::int64_t uid() const {
        return uid_;
    }

    ConstUserType userType() const {
        return userType_;
    }

    auto requestTimeout() const {
        return timeout_;
    }

    auto maxRetriesNumber() const {
        return maxRetriesNumber_;
    }

    TaskContextPtr context() const {
        return context_;
    }

    const QueryRepository& queryRepository() const {
        return queryRepository_;
    }
};

using ConnectionAdaptor = collie::services::db::ConnectionAdaptor<ConnectionContext, Connection>;

template <class Handler>
struct AdaptConnection {
    ConnectionContext context;
    Handler handler;

    template <class T>
    void operator ()(ozo::error_code ec, T&& connection) {
        static_assert(std::is_same_v<std::decay_t<T>, Connection>);
        handler(std::move(ec), ConnectionAdaptor {context, std::forward<T>(connection)});
    }

    using executor_type = decltype(ozo::asio::get_associated_executor(handler));

    executor_type get_executor() const noexcept {
        return ozo::asio::get_associated_executor(handler);
    }

    using allocator_type = decltype(ozo::asio::get_associated_allocator(handler));

    allocator_type get_allocator() const noexcept {
        return ozo::asio::get_associated_allocator(handler);
    }
};

template <class Handler>
auto adaptConnection(const ConnectionContext& context, Handler&& handler) {
    return AdaptConnection<std::decay_t<Handler>> {context, std::forward<Handler>(handler)};
}

} // namespace collie::services::db::contacts
