#pragma once

#include "check_is_user_exists_operation.hpp"
#include "query_repository.hpp"
#include "uid_to_connection_resolver.hpp"
#include "connection_adaptor.hpp"

#include <src/task_context.hpp>

namespace collie::services::db::contacts {

struct WithCheckIsUserExists;
struct WithoutCheckIsUserExists;

template <class UidToConnectionResolver, class CheckIsUserExists = WithCheckIsUserExists>
class ConnectionProvider {
public:
    ConnectionProvider(
        UidToConnectionResolver resolveUid,
        std::int64_t uid,
        ConstUserType userType,
        ozo::time_traits::duration timeout,
        std::size_t maxRetriesNumber,
        QueryRepository queryRepository,
        const TaskContextPtr& context)
        : resolveUid_(std::move(resolveUid)),
          uid_(uid),
          userType_(userType),
          timeout_(timeout),
          maxRetriesNumber_(maxRetriesNumber),
          queryRepository_(queryRepository),
          context_(context) {
    }

    using connection_type = ConnectionAdaptor;

    template <typename Handler>
    void async_get_connection(Handler&& handler) const {
        if constexpr (std::is_same_v<CheckIsUserExists, WithCheckIsUserExists>) {
            resolveUid_(CheckIsUserExistsOperation {
                queryRepository(), uid_, userType_, requestTimeout(),
                adaptConnection(
                    ConnectionContext {uid_, userType_, timeout_, maxRetriesNumber_, queryRepository_, context_},
                    std::forward<Handler>(handler)
                )
            });
        } else {
            resolveUid_(adaptConnection(
                ConnectionContext {uid_, userType_, timeout_, maxRetriesNumber_, queryRepository_, context_},
                std::forward<Handler>(handler)));
        }
    }

    template <typename TimeConstraint, typename Handler>
    void async_get_connection(TimeConstraint, Handler&& handler) const {
        async_get_connection(std::forward<Handler>(handler));
    }

    std::int64_t uid() const {
        return uid_;
    }

    ConstUserType userType() const {
        return userType_;
    }

    auto requestTimeout() const {
        return timeout_;
    }

    auto maxRetriesNumber() const {
        return maxRetriesNumber_;
    }

    TaskContextPtr context() const {
        return context_;
    }

    const QueryRepository& queryRepository() const {
        return queryRepository_;
    }

private:
    UidToConnectionResolver resolveUid_;
    std::int64_t uid_;
    ConstUserType userType_;
    ozo::time_traits::duration timeout_;
    std::size_t maxRetriesNumber_;
    QueryRepository queryRepository_;
    TaskContextPtr context_;
};

} // namespace collie::services::db::contacts
