#pragma once

#include "config.hpp"
#include "connection_provider.hpp"
#include "endpoint_pool.hpp"
#include "query_repository.hpp"
#include "uid_to_connection_resolver.hpp"

#include <src/services/db/timeouts.hpp>
#include <src/services/db/user_id_traits.hpp>
#include <src/services/db/org_user_id.hpp>
#include <src/services/db/passport_user_id.hpp>
#include <src/services/sharpei/sharpei_http_client.hpp>

#include <pgg/service/sharpei_uid_resolver.h>
#include <pgg/service/uid_resolver.h>

namespace collie::services::db::contacts {

template <class CheckIsUserExists = WithCheckIsUserExists>
class MakeConnectionProvider {
public:
    MakeConnectionProvider(const Config& config, GetHttpClient getHttpClient)
        : credentials(config.credentials),
          orgSharpeiClientSettings(config.orgSharpei.client),
          orgSharpeiHttpClient(std::make_shared<const sharpei::SharpeiHttpClient>(
                  getHttpClient, config.orgSharpei.connectTimeout)),
          userSharpeiClientSettings(config.userSharpei.client),
          userSharpeiHttpClient(std::make_shared<const sharpei::SharpeiHttpClient>(
                  std::move(getHttpClient), config.userSharpei.connectTimeout)),
          endpointPool(std::make_shared<EndpointPool>(config.maxTotalPoolsCapacity, config.connectionPool)),
          timeouts(config.timeouts),
          maxRetriesNumber(config.maxRetriesNumber),
          queryRepository(makeQueryRepository(config.queryConf)) {
    }

    template <class T>
    auto operator ()(const TaskContextPtr& context, T userId) const {
        ::sharpei::client::RequestInfo requestInfo;
        requestInfo.requestId = context->requestId();
        requestInfo.clientType = context->clientType();
        requestInfo.userIp = context->userIp();
        requestInfo.uniqId = context->uniq_id();
        const auto makeUidResolver{[&]{
            using UserType = pgg::UidResolver::UserType;
            using UserId = std::decay_t<decltype(userId)>;
            using ::sharpei::client::createSharpeiClient;
            if constexpr (std::is_same_v<UserId, PassportUserId>) {
                auto client{createSharpeiClient(userSharpeiHttpClient, userSharpeiClientSettings,
                        requestInfo)};
                return pgg::createSharpeiUidResolver(credentials, std::move(client), UserType::existing);
            } else if constexpr (std::is_same_v<UserId, OrgUserId>) {
                auto client{createSharpeiClient(orgSharpeiHttpClient, orgSharpeiClientSettings, requestInfo)};
                return pgg::createSharpeiUidResolver(credentials, std::move(client), UserType::organization);
            } else {
                static_assert(std::is_void_v<UserId>, "Invalid uid type");
                return pgg::UidResolverPtr();
            }
        }};

        return ConnectionProvider<UidToConnectionResolver<EndpointPool>, CheckIsUserExists>
            (
            UidToConnectionResolver {
                static_cast<std::int64_t>(userId),
                endpointPool,
                timeouts.connectionPool,
                context,
                makeUidResolver()
            },
            userId,
            UserIdTraits<std::decay_t<decltype(userId)>>::typeName,
            timeouts.request,
            maxRetriesNumber,
            queryRepository,
            context
        );
    }

private:
    pgg::Credentials credentials;
    ::sharpei::client::Settings orgSharpeiClientSettings;
    sharpei::SharpeiHttpClientPtr orgSharpeiHttpClient;
    ::sharpei::client::Settings userSharpeiClientSettings;
    sharpei::SharpeiHttpClientPtr userSharpeiHttpClient;
    std::shared_ptr<EndpointPool> endpointPool;
    Timeouts timeouts;
    std::size_t maxRetriesNumber;
    QueryRepository queryRepository;
};

} // namespace collie::services::db
