#pragma once

#include <src/services/db/error.hpp>
#include <src/get_io_context.hpp>
#include <src/task_context.hpp>

#include <pgg/service/uid_resolver.h>

#include <ozo/connection_pool.h>

namespace collie::services::db::contacts {

template <typename EndpointPool, typename Handler>
struct UidToConnectionResolveOperation {
    std::shared_ptr<EndpointPool> endpoints;
    ozo::connection_pool_timeouts timeouts;
    TaskContextPtr ctx;
    std::decay_t<Handler> handler;

    void perform(pgg::UidResolverPtr uidResolver, std::int64_t uid) const {
        uidResolver->asyncGetConnInfo(pgg::UidResolveParams(std::to_string(uid)), *this);
    }

    template <typename Range>
    void operator () (error_code ec, Range&& connInfo) {
        using ConnectionPool = typename EndpointPool::ConnectionPool;
        using Connection = ozo::connection_type<ConnectionPool>;

        if (ec) {
            return handler(ec.base(), Connection{});
        }

        if (connInfo.empty()) {
            return handler(ozo::error_code(Error::userShardUnavailable), Connection{});
        }

        LOGDOG_(ctx->logger(), debug, log::message="use conn info from sharpei",
                log::conn_info=connInfo.front());

        auto pool = endpoints->getConnectionPool(connInfo.front());
        if (!pool) {
            return handler(ozo::error_code(Error::endpointsOverflow), Connection{});
        }

        using ozo::async_get_connection;
        async_get_connection(
            (*pool)[getIoContext(ctx->yield())],
            timeouts.connect,
            std::move(handler)
        );
    }

    using executor_type = decltype(boost::asio::get_associated_executor(handler));

    executor_type get_executor() const noexcept {
        return boost::asio::get_associated_executor(handler);
    }

    using allocator_type = decltype(boost::asio::get_associated_allocator(handler));

    allocator_type get_allocator() const noexcept {
        return boost::asio::get_associated_allocator(handler);
    }
};

} // namespace collie::services::db::contacts
