#pragma once

#include "connection_pool.hpp"

#include <ozo/ext/std/shared_ptr.h>

#include <boost/range/algorithm/find_if.hpp>

#include <mutex>

namespace collie::services::db {

template <typename OidMap = ozo::empty_oid_map>
class EndpointPool {
public:
    using ConnectionPool = db::ConnectionPool<OidMap>;
    using ConnectionPoolPtr = std::shared_ptr<ConnectionPool>;

    EndpointPool(const std::size_t maxTotalPoolsCapacity, const ozo::connection_pool_config& poolConfig)
            : maxTotalPoolsCapacity(maxTotalPoolsCapacity), poolConfig(poolConfig) {
        if (maxTotalPoolsCapacity < poolConfig.capacity) {
            throw std::logic_error("max total pools capacity ("
                + std::to_string(maxTotalPoolsCapacity) + ") is less than pool capacity ("
                + std::to_string(poolConfig.capacity) + ")");
        }
    }

    ConnectionPoolPtr getConnectionPool(const std::string& conninfo) {
        const std::lock_guard<std::mutex> lock(mutex);
        auto it = connectionPools.find(conninfo);
        if (it == connectionPools.end()) {
            if (willOverflow() && !discardUnusedConnectionPool()) {
                return nullptr;
            }
            it = connectionPools.emplace(
                conninfo,
                std::make_shared<ConnectionPool>(
                    ozo::make_connection_pool(ozo::connection_info<OidMap>(conninfo), poolConfig)
                )
            ).first;
        }
        return it->second;
    }

private:
    std::mutex mutex;
    std::size_t maxTotalPoolsCapacity;
    ozo::connection_pool_config poolConfig;
    std::map<std::string, ConnectionPoolPtr> connectionPools;

    bool willOverflow() const {
        return poolConfig.capacity * (connectionPools.size() + 1) > maxTotalPoolsCapacity;
    }

    bool discardUnusedConnectionPool() {
        const auto unused = boost::find_if(connectionPools,
            [&] (const auto& v) { return v.second.use_count() == 1; });
        if (unused == connectionPools.end()) {
            return false;
        }
        connectionPools.erase(unused);
        return true;
    }
};

} // namespace collie::services::db
