#pragma once

#include <src/error_code.hpp>

namespace collie::services::db {

enum class Error {
    ok,
    userShardUnavailable,
    endpointsOverflow,
    databaseError,
    shardMasterUnavailable,
};

} // namespace collie::services::db

namespace boost::system {

template <>
struct is_error_code_enum<collie::services::db::Error> : std::true_type {};

} // namespace boost::system

namespace collie::services::db {

class ErrorCategory final : public boost::system::error_category {
public:
    const char* name() const noexcept override {
        return "collie::services::db::ErrorCategory";
    }

    std::string message(int value) const override {
        switch (static_cast<Error>(value)) {
            case Error::ok:
                return "ok";
            case Error::userShardUnavailable:
                return "database shard for this user is not available";
            case Error::endpointsOverflow:
                return "limit of databases connections has been reached";
            case Error::databaseError:
                return "database error";
            case Error::shardMasterUnavailable:
                return "database shard master is not available";
        }
        return "unknown error code: " + std::to_string(value);
    }

    static const ErrorCategory& instance() {
        static const ErrorCategory errorCategory;
        return errorCategory;
    }
};

inline error_code::base_type make_error_code(Error ec) {
    return error_code::base_type(static_cast<int>(ec), ErrorCategory::instance());
}

} // namespace collie::services::db
