#pragma once

#include "connection_provider_config.hpp"

#include <src/task_context.hpp>
#include <src/services/db/connection_adapter.hpp>

#include <ozo/connection_pool.h>

namespace collie::services::db::events_queue {

using Connection = ozo::connection_type<ozo::connection_pool<ozo::connection_info<OidMap>>>;

struct ConnectionContext {
    ConnectionProviderConfig config_;
    TaskContextPtr context_;

    auto requestTimeout() const {
        return config_.timeouts.request;
    }

    auto maxRetriesNumber() const {
        return config_.maxRetriesNumber;
    }

    TaskContextPtr context() const {
        return context_;
    }

    const QueryRepository& queryRepository() const {
        return config_.queryRepository;
    }
};

using ConnectionAdaptor = collie::services::db::ConnectionAdaptor<ConnectionContext, Connection>;

template <class Handler>
struct AdaptConnection {
    ConnectionContext context;
    Handler handler;

    template <class T>
    void operator ()(ozo::error_code ec, T&& connection) {
        static_assert(std::is_same_v<std::decay_t<T>, Connection>);
        handler(std::move(ec), ConnectionAdaptor {context, std::forward<T>(connection)});
    }

    using executor_type = decltype(ozo::asio::get_associated_executor(handler));

    executor_type get_executor() const noexcept {
        return ozo::asio::get_associated_executor(handler);
    }

    using allocator_type = decltype(ozo::asio::get_associated_allocator(handler));

    allocator_type get_allocator() const noexcept {
        return ozo::asio::get_associated_allocator(handler);
    }
};

template <class Handler>
auto adaptConnection(const ConnectionContext& context, Handler&& handler) {
    return AdaptConnection<std::decay_t<Handler>> {context, std::forward<Handler>(handler)};
}

} // namespace collie::services::db::events_queue
