#pragma once

#include "query.hpp"

#include <src/services/db/connection_provider.hpp>
#include <src/services/db/utils.hpp>

#include <ozo/request.h>
#include <ozo/shortcuts.h>

#include <optional>

namespace collie::services::db::events_queue {

template <class P>
class GetUserToSync {
    static_assert(ConnectionProvider<P>);

public:
    GetUserToSync(P provider)
        : provider(std::move(provider)) {}

    template <class C>
    expected<std::optional<UserToSync>> operator ()(
        C& connection,
        std::int64_t syncIntervalInMinutes) const
    {
        using ozo::request;
        using ozo::to_const_char;
        using ozo::get_text;
        using ozo::get_error_context;

        const auto context = provider.context();
        const auto& repository = provider.queryRepository();
        const auto queryByCount = repository.template make_query<query::GetUserToSyncByEventsCount>();
        std::vector<UserToSync> usersToSync;
        ozo::error_code ec;
        usersToSync.reserve(1);
        connection = request(connection, queryByCount, ozo::into(usersToSync), context->yield()[ec]);
        if (ec) {
            LOGDOG_(context->logger(), error,
                log::error_code=ec,
                log::query=to_const_char(get_text(queryByCount)),
                log::message=(connection ? get_error_context(connection) : std::string())
            );
            return make_unexpected(error_code(Error::databaseError));
        }

        if (usersToSync.empty()) {
            const auto queryByDate = repository.template make_query<query::GetUserToSyncByLastSyncDate>(
                {syncIntervalInMinutes});
            connection = request(connection, queryByDate, ozo::into(usersToSync), context->yield()[ec]);
            if (ec) {
                LOGDOG_(context->logger(), error,
                    log::error_code=ec,
                    log::query=to_const_char(get_text(queryByDate)),
                    log::message=(connection ? get_error_context(connection) : std::string())
                );
                return make_unexpected(error_code(Error::databaseError));
            }
        }

        if (usersToSync.empty()) {
            return {};
        }
        return {std::move(usersToSync.front())};
    }

private:
    P provider;
};

} // namespace collie::services::db::events_queue
