#pragma once

#include <src/services/db/error.hpp>
#include <src/get_io_context.hpp>
#include <src/task_context.hpp>

#include <pgg/service/shard_resolver.h>

#include <ozo/connection_pool.h>

namespace collie::services::db::events_queue {

template <typename EndpointPool, typename Handler>
struct MasterResolveOperation {
    std::shared_ptr<EndpointPool> endpoints;
    ozo::connection_pool_timeouts timeouts;
    TaskContextPtr context;
    Handler handler;

    void perform(pgg::ShardResolverPtr shardResolver, const ::sharpei::client::Shard::Id& shardId) const {
        pgg::ShardResolveParams params(shardId);
        params.endpointType(pgg::query::Traits::EndpointType::master);
        shardResolver->asyncGetConnInfo(params, *this);
    }

    template <typename Range>
    void operator () (error_code ec, Range&& connInfo) {
        using ConnectionPool = typename EndpointPool::ConnectionPool;
        using Connection = ozo::connection_type<ConnectionPool>;

        if (ec) {
            return handler(ec.base(), Connection{});
        }

        if (connInfo.empty()) {
            return handler(ozo::error_code(Error::shardMasterUnavailable), Connection{});
        }

        LOGDOG_(context->logger(), debug, log::message="use conn info from sharpei",
                log::conn_info=connInfo.front());

        auto pool = endpoints->getConnectionPool(connInfo.front());
        if (!pool) {
            return handler(ozo::error_code(Error::endpointsOverflow), Connection{});
        }

        using ozo::async_get_connection;
        async_get_connection(
            (*pool)[getIoContext(context->yield())],
            timeouts.connect,
            std::forward<Handler>(handler)
        );
    }

    using executor_type = decltype(boost::asio::get_associated_executor(handler));

    executor_type get_executor() const noexcept {
        return boost::asio::get_associated_executor(handler);
    }

    using allocator_type = decltype(boost::asio::get_associated_allocator(handler));

    allocator_type get_allocator() const noexcept {
        return boost::asio::get_associated_allocator(handler);
    }
};

} // namespace collie::services::db::events_queue
