#pragma once

#include "connection_provider.hpp"
#include "error.hpp"
#include "retry.hpp"
#include "utils.hpp"

#include <src/expected.hpp>
#include <src/log.hpp>

#include <ozo/execute.h>
#include <ozo/shortcuts.h>

namespace collie::services::db {

namespace adl {

template<typename T> struct ExecuteImpl {
    template<typename... Ts> static auto apply(Ts&& ...v) {
        return ozo::execute(std::forward<Ts>(v)...);
    }
};

template <typename T, typename... Ts>
auto execute(T&& v, Ts&& ...vs) {
    return ExecuteImpl<std::decay_t<T>>::apply(std::forward<T>(v), std::forward<Ts>(vs)...);
}

}

template <class T>
struct ExecuteImpl {
    template <class Query>
    static auto apply(T provider, Query query) -> expected<ozo::connection_type<T>> {
        using ozo::error_message;
        using ozo::get_error_context;
        using ozo::get_text;
        using adl::execute;
        using ozo::to_const_char;

        static_assert(ConnectionProvider<decltype(db::unwrap(provider))>);

        const auto requestTimeout = db::unwrap(provider).requestTimeout();
        const auto context = db::unwrap(provider).context();

        LOGDOG_(context->logger(), notice, log::query=to_const_char(get_text(query)));

        ozo::error_code ec;
        const auto conn = execute(
            provider,
            query,
            requestTimeout,
            context->yield()[ec]
        );

        if (!ec) {
            return conn;
        }

        LOGDOG_(context->logger(), error,
            log::error_code=ec,
            log::query=to_const_char(get_text(query)),
            log::message=(!ozo::is_null_recursive(conn) ? get_error_context(conn) : std::string()),
            log::pq_message=(!ozo::is_null_recursive(conn) ? error_message(conn) : std::string_view())
        );

        if (!userNotFound(ec)) {
            ec = Error::databaseError;
        }

        return make_unexpected(error_code{std::move(ec)});
    }
};

struct ReturnConnection {};

template <class T, class Query>
auto execute(T&& provider, Query&& query, ReturnConnection) {
    return ExecuteImpl<std::decay_t<T>>::apply(
        std::forward<T>(provider),
        std::forward<Query>(query)
    );
}

template <class T>
struct ExecuteWithQueryRepositoryImpl {
    template <class Query>
    static expected<void> apply(T provider, Query&& params) {
        using Q = std::decay_t<Query>;

        static_assert(ConnectionProvider<decltype(db::unwrap(provider))>);
        static_assert(std::is_same_v<Q, typename Q::parameters_type>);

        const auto query = db::unwrap(provider).queryRepository()
            .template make_query<Q>(std::forward<Query>(params));

        return execute(std::move(provider), query, ReturnConnection{}).bind([&](auto&&){});
    }
};

template <class T, class Query>
auto execute(T&& provider, Query&& query) {
    return ExecuteWithQueryRepositoryImpl<std::decay_t<T>>::apply(
        std::forward<T>(provider),
        std::forward<Query>(query)
    );
}

template <class T>
struct ExecuteImpl<Retry<T>> {
    template <class Query>
    static auto apply(Retry<T> provider, Query query) -> expected<ozo::connection_type<T>> {
        static_assert(ConnectionProvider<T>);

        const auto maxRetriesNumber = db::unwrap(provider).maxRetriesNumber();
        std::size_t tryNumber = 0;

        while (true) {
            const auto result = execute(db::unwrap(provider), query, ReturnConnection{});

            if (result) {
                return result;
            }

            if (!isRetriable(result.error()) || ++tryNumber > maxRetriesNumber) {
                return result;
            }
        }
    }
};

} // namespace collie::services::db
