#pragma once

#include "connection_provider.hpp"

#include <src/error_code.hpp>
#include <src/expected.hpp>
#include <src/log.hpp>

#include <ozo/connection.h>

namespace collie::services::db {

template <class T>
struct GetConnectionImpl {
    static auto apply(T provider) -> expected<ozo::connection_type<T>> {
        using ozo::get_connection;
        using ozo::get_error_context;
        using ozo::error_message;

        static_assert(ConnectionProvider<T>);

        ozo::error_code ec;
        const auto yield = provider.context()->yield();
        const auto connection = get_connection(provider, yield[ec]);

        if (ec) {
            LOGDOG_(provider.context()->logger(), error,
                log::error_code=ec,
                log::message=(!ozo::is_null_recursive(connection) ? get_error_context(connection) : std::string()),
                log::pq_message=(!ozo::is_null_recursive(connection) ? error_message(connection) : std::string_view())
            );
            return make_unexpected(error_code(ec));
        }

        return connection;
    }
};

template <typename T>
auto getConnection(T&& provider) {
    return GetConnectionImpl<std::decay_t<T>>::apply(std::forward<T>(provider));
}

} // namespace collie::services::db
