#pragma once

#include "connection_provider.hpp"
#include "error.hpp"
#include "retry.hpp"
#include "utils.hpp"

#include <src/expected.hpp>
#include <src/log.hpp>

#include <ozo/request.h>
#include <ozo/shortcuts.h>

namespace collie::services::db {

namespace adl {

template <typename T>
struct RequestImpl {
    template <typename ...Ts>
    static auto apply(Ts&& ...v) { return ozo::request(std::forward<Ts>(v)...);}
};

template <class T, class ...Ts>
auto request(T&& v, Ts&& ...vs) {
    return RequestImpl<std::decay_t<T>>::apply(std::forward<T>(v), std::forward<Ts>(vs)...);
}

} // namespace adl

template <class T>
struct RequestImpl {
    template <class Query, class Out>
    static auto apply(T provider, Query query, Out out) -> expected<ozo::connection_type<T>> {
        using ozo::error_message;
        using ozo::get_error_context;
        using ozo::get_text;
        using adl::request;
        using ozo::to_const_char;

        static_assert(ConnectionProvider<decltype(db::unwrap(provider))>);

        const auto requestTimeout = db::unwrap(provider).requestTimeout();
        const auto context = db::unwrap(provider).context();

        LOGDOG_(context->logger(), notice, log::query=to_const_char(get_text(query)));

        ozo::error_code ec;
        const auto conn = request(
            provider,
            query,
            requestTimeout,
            out,
            context->yield()[ec]
        );

        if (!ec) {
            return conn;
        }

        LOGDOG_(context->logger(), error,
            log::error_code=ec,
            log::query=to_const_char(get_text(query)),
            log::message=(!ozo::is_null_recursive(conn) ? get_error_context(conn) : std::string()),
            log::pq_message=(!ozo::is_null_recursive(conn) ? error_message(conn) : std::string_view())
        );

        if (!userNotFound(ec)) {
            ec = Error::databaseError;
        }

        return make_unexpected(error_code{std::move(ec)});
    }
};

template <class T, class Query, class Out>
auto request(T&& provider, Query&& query, Out&& out) {
    return RequestImpl<std::decay_t<T>>::apply(
        std::forward<T>(provider),
        std::forward<Query>(query),
        std::forward<Out>(out)
    );
}

template <class T>
struct RequestWithQueryRepositoryImpl {
    template <class Query>
    static auto apply(T provider, Query&& params)
            -> expected<std::vector<typename std::decay_t<Query>::result_type>> {
        using Q = std::decay_t<Query>;

        static_assert(ConnectionProvider<decltype(db::unwrap(provider))>);
        static_assert(std::is_same_v<Q, typename Q::parameters_type>);

        using Result = std::vector<typename Q::result_type>;

        const auto query = db::unwrap(provider).queryRepository()
            .template make_query<Q>(std::forward<Query>(params));
        Result result;
        return request(std::move(provider), query, ozo::into(result))
            .bind([&] (auto&&) { return result; });
    }
};

template <class T, class Query>
auto request(T&& provider, Query&& params) {
    return RequestWithQueryRepositoryImpl<std::decay_t<T>>::apply(
        std::forward<T>(provider),
        std::forward<Query>(params)
    );
}

template <class T>
struct RequestImpl<Retry<T>> {
    template <class Query, class Out>
    static auto apply(Retry<T> provider, Query query, Out out) -> expected<ozo::connection_type<T>> {
        static_assert(ConnectionProvider<T>);

        const auto maxRetriesNumber = db::unwrap(provider).maxRetriesNumber();
        std::size_t tryNumber = 0;

        while (true) {
            const auto result = request(db::unwrap(provider), query, out);

            if (result) {
                return result;
            }

            if (!isRetriable(result.error()) || ++tryNumber > maxRetriesNumber) {
                return result;
            }
        }
    }
};

} // namespace collie::services::db
