#include "directory_client_impl.hpp"
#include "error.hpp"
#include "utils.hpp"

#include <butil/http/headers.h>
#include <src/services/retry.hpp>

#include <sstream>

BOOST_FUSION_ADAPT_STRUCT(collie::services::directory::Event,
    name,
    timestamp,
    object_type,
    id,
    revision
)

BOOST_FUSION_ADAPT_STRUCT(collie::services::directory::Name,
    first,
    last
)

BOOST_FUSION_ADAPT_STRUCT(collie::services::directory::Contact,
    type,
    value
)

BOOST_FUSION_ADAPT_STRUCT(collie::services::directory::User::Department,
    id
)

BOOST_FUSION_ADAPT_STRUCT(collie::services::directory::User,
    id,
    name,
    contacts,
    birthday,
    department,
    position
)

BOOST_FUSION_ADAPT_STRUCT(collie::services::directory::Parent,
    id
)

BOOST_FUSION_ADAPT_STRUCT(collie::services::directory::Department,
    id,
    description,
    email,
    parent,
    name
)

BOOST_FUSION_ADAPT_STRUCT(collie::services::directory::Group,
    id,
    description,
    email,
    name,
    type
)

BOOST_FUSION_ADAPT_STRUCT(collie::services::directory::OrgInfo,
    name
)

BOOST_FUSION_DEFINE_STRUCT((collie)(services)(directory), EventsResponse,
    (collie::services::directory::Links, links)
    (collie::services::directory::Events, result)
)

BOOST_FUSION_DEFINE_STRUCT((collie)(services)(directory), UsersResponse,
    (collie::services::directory::Links, links)
    (collie::services::directory::Users, result)
)

BOOST_FUSION_DEFINE_STRUCT((collie)(services)(directory), DepartmentsResponse,
    (collie::services::directory::Links, links)
    (collie::services::directory::Departments, result)
)

BOOST_FUSION_DEFINE_STRUCT((collie)(services)(directory), GroupsResponse,
    (collie::services::directory::Links, links)
    (collie::services::directory::Groups, result)
)

BOOST_FUSION_ADAPT_STRUCT(collie::services::directory::Links,
    next,
    last
)

BOOST_FUSION_ADAPT_STRUCT(collie::services::directory::Organization,
    id
)

BOOST_FUSION_ADAPT_STRUCT(collie::services::directory::OrganizationsResponse,
    links,
    result
)

BOOST_FUSION_DEFINE_STRUCT((collie)(services)(directory), DirectoryError,
    (std::string, message)
    (std::string, code)
)

namespace collie::services::directory {

namespace {

auto makeRetryCondition(std::size_t retries) {
    return makeCompositeRetryCondition(
        retry_condition::LimitedRetries {retries},
        retry_condition::AcceptHttp404 {},
        retry_condition::RetryHttp5xx {}
    );
}

template<typename Values, typename GetPage> expected<Values> getAllPages(std::string url, GetPage&& getPage) {
    Values values;
    for (;;) {
        auto result{getPage(url)};
        if (!result) {
            return make_unexpected(std::move(result).error());
        }

        auto& resultValue{result.value()};
        values.reserve(values.size() + resultValue.values.size());
        std::move(resultValue.values.begin(), resultValue.values.end(), std::back_inserter(values));
        if (resultValue.next) {
            url = *std::move(resultValue.next);
        } else {
            break;
        }
    }

    return values;
}

template <class Response, class Page>
struct HandleRequest {
    TaskContextPtr context;
    expected<Page> operator ()(yhttp::response&& response) const {
        const std::string source{"Directory"};
        if (response.status == 404) {
            return fromJson<DirectoryError, Error>(response.body, context, source).bind(
                    [&](const auto& error) {
                if (error.code == "organization_deleted") {
                    return make_expected_from_error<Page>(error_code{Error::organizationDeleted, error.message});
                }
                return make_expected_from_error<Page>(error_code{Error::organizationNotFound, error.message});
            });
        } else if (response.status != 200) {
            return make_unexpected(error_code{services::Error::httpError});
        } else {
            return fromJson<Response, Error>(response.body, context, source).bind([&](auto&& parsed){
                return Page{std::move(parsed.result), std::move(parsed.links.next),
                        std::move(parsed.links.last)};});
        }
    }
};

template<typename Response> struct HandleNonpagedRequest {
    expected<Response> operator()(yhttp::response&& response) const {
        const std::string source{"Directory"};
        if ((response.status == 403) || (response.status == 404)) {
            const std::unordered_map<decltype(response.status), Error> errorCodes {
                {403, Error::requestForbidden},
                {404, Error::organizationNotFound}
            };

            return fromJson<DirectoryError, Error>(response.body, context, source).bind(
                    [&](const auto& error) {
                return make_expected_from_error<Response>(
                        error_code{errorCodes.at(response.status), error.message});
            });
        } else if (response.status != 200) {
            return make_unexpected(error_code{services::Error::httpError});
        } else {
            return fromJson<Response, Error>(response.body, context, source).bind([&](auto&& parsed){
                return std::move(parsed);});
        }
    }

    TaskContextPtr context;
};

}

DirectoryClientImpl::DirectoryClientImpl(const Config config, const GetHttpClient& getHttpClient,
        const GetTvm2Module& getTvm2Module)
    : config(config),
      getHttpClient(getHttpClient),
      getTvm2Module(getTvm2Module) {
}

expected<std::optional<Event>> DirectoryClientImpl::getLastEvent(const TaskContextPtr& context,
        const GetLastEventParams& params) const {
    return getTvmServiceTicket(context).bind([&](const auto& serviceTicket){
            return getLastEvent(context, params, serviceTicket);});
}

expected<OrgInfo> DirectoryClientImpl::getOrgInfo(const TaskContextPtr& context, OrgId orgId) const {
    return getTvmServiceTicket(context).bind([&](const auto& serviceTicket){
            return getOrgInfo(context, orgId, serviceTicket);});
}

expected<Users> DirectoryClientImpl::getUsers(const TaskContextPtr& context, OrgId orgId) const {
    return getTvmServiceTicket(context)
        .bind([&] (const auto& serviceTicket) { return getUsers(context, orgId, serviceTicket); });
}

expected<Departments>  DirectoryClientImpl::getDepartments(const TaskContextPtr& context, OrgId orgId) const {
    return getTvmServiceTicket(context)
        .bind([&] (const auto& serviceTicket) { return getDepartments(context, orgId, serviceTicket); });
}

expected<Groups> DirectoryClientImpl::getGroups(const TaskContextPtr& context, OrgId orgId) const {
    return getTvmServiceTicket(context)
        .bind([&] (const auto& serviceTicket) { return getGroups(context, orgId, serviceTicket); });
}

expected<std::optional<Event>> DirectoryClientImpl::getLastEvent(const TaskContextPtr& context,
        const GetLastEventParams& params, const std::string& serviceTicket) const {
    std::ostringstream firstPageUrlStream;
    firstPageUrlStream << config.location << "/v9/events/?per_page=10&revision__gt=" << std::max(
            static_cast<std::int64_t>(params.syncedRevision), static_cast<std::int64_t>(1)) - 1;
    std::string firstPageUrl {firstPageUrlStream.str()};
    auto result{getEvents(context, params.orgId, serviceTicket, firstPageUrl)};
    if (!result) {
        return make_unexpected(std::move(result).error());
    }

    auto& firstPageResult{result.value()};
    Events events;
    events.reserve(firstPageResult.values.size());
    std::move(firstPageResult.values.begin(), firstPageResult.values.end(), std::back_inserter(events));
    if (events.size() && firstPageResult.last && firstPageUrl != *firstPageResult.last) {
        auto result{getEvents(context, params.orgId, serviceTicket, *firstPageResult.last)};
        if (!result) {
            return make_unexpected(std::move(result).error());
        }

        auto& lastPageResult{result.value()};
        events.reserve(events.size() + lastPageResult.values.size());
        std::move(lastPageResult.values.begin(), lastPageResult.values.end(), std::back_inserter(events));
    }
    if (events.size()) {
        if (!std::is_sorted(events.begin(), events.end(), eventLessByRevisionAndId)) {
            std::sort(events.begin(), events.end(), eventLessByRevisionAndId);
        }

        return std::make_optional(std::move(events.back()));
    }

    return std::optional<Event>{};
}

expected<OrgInfo> DirectoryClientImpl::getOrgInfo(const TaskContextPtr& context, OrgId orgId,
        const std::string& serviceTicket) const {
    using namespace http_getter;
    std::string url{config.location + "/v11/organizations/" + std::to_string(orgId) + "/?fields=name"};
    const auto request = get(std::move(url)).
            headers(requestId=context->requestId(), http_getter::serviceTicket=serviceTicket).
            timeouts(config.httpOptions.timeouts.total, config.httpOptions.timeouts.connect).
            make();
    return perform(context, request).bind(HandleNonpagedRequest<OrgInfo>{context});
}

expected<Users> DirectoryClientImpl::getUsers(const TaskContextPtr& context, OrgId orgId,
        const std::string& serviceTicket) const {
    std::string initialUrl{config.location +
            "/v9/users/?per_page=1000&fields=id,name,contacts,aliases,birthday,department,position"};
    return getAllPages<Users>(std::move(initialUrl), [&](auto&& url){return getUsers(
            context, orgId, serviceTicket, std::move(url));});
}

expected<Departments> DirectoryClientImpl::getDepartments(const TaskContextPtr& context, OrgId orgId,
        const std::string& serviceTicket) const {
    std::string initialUrl{config.location +
            "/v9/departments/?per_page=1000&fields=id,description,email,parent,name"};
    return getAllPages<Departments>(std::move(initialUrl), [&](auto&& url){return getDepartments(
            context, orgId, serviceTicket, std::move(url));});
}

expected<Groups> DirectoryClientImpl::getGroups(const TaskContextPtr& context, OrgId orgId,
        const std::string& serviceTicket) const {
    std::string initialUrl{config.location +
            "/v9/groups/?per_page=1000&fields=id,description,email,name,type"};
    return getAllPages<Groups>(std::move(initialUrl), [&](auto&& url){return getGroups(
            context, orgId, serviceTicket, std::move(url));});
}

expected<EventsPage> DirectoryClientImpl::getEvents(const TaskContextPtr& context, OrgId orgId,
        const std::string& serviceTicket, std::string url) const {
    using namespace http_getter;
    const auto request = get(std::move(url))
                        .headers(requestId=context->requestId(), http_getter::serviceTicket=serviceTicket, "X-ORG-ID"_hdr=std::to_string(orgId))
                        .timeouts(config.httpOptions.timeouts.total, config.httpOptions.timeouts.connect)
                        .make();
    return perform(context, request).bind(HandleRequest<EventsResponse, EventsPage>{context});
}

expected<UsersPage> DirectoryClientImpl::getUsers(const TaskContextPtr& context, OrgId orgId,
        const std::string& serviceTicket, std::string url) const {
    using namespace http_getter;
    const auto request = get(std::move(url))
                        .headers(requestId=context->requestId(), http_getter::serviceTicket=serviceTicket, "X-ORG-ID"_hdr=std::to_string(orgId))
                        .timeouts(config.httpOptions.timeouts.total, config.httpOptions.timeouts.connect)
                        .make();
    return perform(context, request).bind(HandleRequest<UsersResponse, UsersPage>{context});
}

expected<DepartmentsPage> DirectoryClientImpl::getDepartments(const TaskContextPtr& context, OrgId orgId,
        const std::string& serviceTicket, std::string url) const {
    using namespace http_getter;
    const auto request = get(std::move(url))
                        .headers(requestId=context->requestId(), http_getter::serviceTicket=serviceTicket, "X-ORG-ID"_hdr=std::to_string(orgId))
                        .timeouts(config.httpOptions.timeouts.total, config.httpOptions.timeouts.connect)
                        .make();
    return perform(context, request).bind(HandleRequest<DepartmentsResponse, DepartmentsPage>{context});
}

expected<GroupsPage> DirectoryClientImpl::getGroups(const TaskContextPtr& context, OrgId orgId,
        const std::string& serviceTicket, std::string url) const {
    using namespace http_getter;
    const auto request = get(std::move(url))
                        .headers(requestId=context->requestId(), http_getter::serviceTicket=serviceTicket, "X-ORG-ID"_hdr=std::to_string(orgId))
                        .timeouts(config.httpOptions.timeouts.total, config.httpOptions.timeouts.connect)
                        .make();
    return perform(context, request).bind(HandleRequest<GroupsResponse, GroupsPage>{context});
}

expected<Organizations> DirectoryClientImpl::getOrganizations(const TaskContextPtr& context) const {
    return getTvmServiceTicket(context)
        .bind([&] (const auto& serviceTicket) -> expected<Organizations> {
            Organizations organizations;
            std::ostringstream url_stream;
            url_stream << config.location << "/v9/organizations/?fields=id&per_page=1000";
            std::string url {url_stream.str()};
            bool hasNextPage {false};
            do {
                auto result = getOrganizations(context, serviceTicket, url);
                if (!result) {
                    return make_unexpected(result.error());
                }
                auto& pageIds = result.value().result;
                std::ostringstream message;
                message
                    << "get org_ids size=" << std::to_string(pageIds.size())
                    << " start_id=" <<  std::to_string(pageIds.begin()->id)
                    << " end_id=" << std::to_string(pageIds.rbegin()->id);
                LOGDOG_(context->logger(), notice, log::message=message.str());
                if (pageIds.size()) {
                    organizations.insert(organizations.end(), pageIds.begin(), pageIds.end());
                }
                hasNextPage = false;
                if (result.value().links.next) {
                    url = std::move(*result.value().links.next);
                    hasNextPage = true;
                }
            } while (hasNextPage);
            return organizations;
        });
}

expected<OrganizationsResponse> DirectoryClientImpl::getOrganizations(
        const TaskContextPtr& context, const std::string& serviceTicket, const std::string& url) const {
    using namespace http_getter;
    const auto request = get(url)
                        .headers(requestId=context->requestId(), http_getter::serviceTicket=serviceTicket)
                        .timeouts(config.httpOptions.timeouts.total, config.httpOptions.timeouts.connect)
                        .make();
    return perform(context, request)
        .bind([&] (const auto& response) -> expected<OrganizationsResponse> {
            if (response.status != 200) {
                return make_unexpected(error_code(services::Error::httpError));
            } else {
                const std::string source{"Directory"};
                auto parsedResponse = fromJson<OrganizationsResponse, Error>(response.body, context, source);
                if (!parsedResponse) {
                    return make_unexpected(parsedResponse.error());
                }
                return parsedResponse.value();
            }
        });
}

expected<std::string> DirectoryClientImpl::getTvmServiceTicket(const TaskContextPtr& context) const {
    std::string result;
    const auto ec = getTvm2Module()->get_service_ticket(config.targetServiceName, result);
    if (ec) {
        LOGDOG_(context->logger(), error,
            log::message="failed to get TVM2 service ticket for directory request",
            log::error_code=ec
        );
        return make_unexpected(error_code(Error::tvmServiceTicketError, ec.message()));
    }
    return result;
}

expected<yhttp::response> DirectoryClientImpl::perform(const TaskContextPtr& context,
        const http_getter::Request& request) const {
    return performWithRetries(*getHttpClient(), context, request, makeRetryCondition(config.retries))
        .catch_error([] (auto ec) {
            if (ec == services::Error::httpError) {
                return make_unexpected(error_code(
                    services::Error::httpError,
                    "request to directory failed after all retries"
                ));
            } else {
                return make_unexpected(std::move(ec));
            }
        });
}

} // namespace collie::services::directory
