#pragma once

#include <src/error_code.hpp>

namespace collie::services::directory {

enum class Error {
    ok,
    responseParseError,
    requestForbidden,
    organizationNotFound,
    tvmServiceTicketError,
    organizationDeleted
};

} // namespace collie::services::directory

namespace boost::system {

template <>
struct is_error_code_enum<collie::services::directory::Error> : std::true_type {};

} // namespace boost::system

namespace collie::services::directory {

class ErrorCategory final : public boost::system::error_category {
public:
    const char* name() const noexcept override {
        return "collie::services::directory::ErrorCategory";
    }

    std::string message(int value) const override {
        switch (static_cast<Error>(value)) {
            case Error::ok:
                return "ok";
            case Error::responseParseError:
                return "response parse error";
            case Error::requestForbidden:
                return "request is forbidden";
            case Error::organizationNotFound:
                return "organization is not found";
            case Error::tvmServiceTicketError:
                return "can't get tvm service ticket";
            case Error::organizationDeleted:
                return "organization was deleted";
        }
        return "unknown error code: " + std::to_string(value);
    }

    static const ErrorCategory& instance() {
        static const ErrorCategory errorCategory;
        return errorCategory;
    }
};

inline error_code::base_type make_error_code(Error ec) {
    return error_code::base_type(static_cast<int>(ec), ErrorCategory::instance());
}

} // namespace collie::services::directory
