#pragma once

#include <src/error_code.hpp>

namespace collie::services::ml {

enum class Error {
    ok,
    responseParseError,
    tvmServiceTicketError
};

} // namespace collie::services::ml

namespace boost::system {

template<> struct is_error_code_enum<collie::services::ml::Error> : std::true_type {
};

} // namespace boost::system

namespace collie::services::ml {

class ErrorCategory final : public boost::system::error_category {
public:
    virtual const char* name() const noexcept override {
        return "collie::services::ml::ErrorCategory";
    }

    virtual std::string message(int value) const override {
        switch (static_cast<Error>(value)) {
            case Error::ok:
                return "ok";
            case Error::responseParseError:
                return "response parse error";
            case Error::tvmServiceTicketError:
                return "can't get tvm service ticket";
        }

        return "unknown error code: " + std::to_string(value);
    }

    static const ErrorCategory& instance() {
        static const ErrorCategory errorCategory;
        return errorCategory;
    }
};

inline error_code::base_type make_error_code(Error ec) {
    return error_code::base_type(static_cast<int>(ec), ErrorCategory::instance());
}

} // namespace collie::services::ml
