#include "ml_client_impl.hpp"
#include "error.hpp"

#include <src/services/retry.hpp>
#include <butil/http/headers.h>

BOOST_FUSION_ADAPT_STRUCT(collie::services::ml::MlEntry,
    id,
    email,
    name
)

BOOST_FUSION_ADAPT_STRUCT(collie::services::ml::MlEntriesResponse,
    maillists
)

namespace collie::services::ml {

namespace {

auto makeRetryCondition(std::size_t retries) {
    return makeCompositeRetryCondition(retry_condition::LimitedRetries{retries},
            retry_condition::RetryHttp5xx{});
}

} // namespace

MlClientImpl::MlClientImpl(const Config& config, const GetHttpClient& getHttpClient,
        const GetTvm2Module& getTvm2Module)
        : config(config)
        , getHttpClient(getHttpClient)
        , getTvm2Module(getTvm2Module) {
}

expected<MlEntries> MlClientImpl::getMlEntries(const TaskContextPtr& context) {
    const auto getEntries{[&](const auto& ticket){return getMlEntriesImpl(context, ticket);}};
    return getTvmServiceTicket(context).bind(getEntries).bind([&](auto&& response){
            return std::move(response.maillists);});
}

expected<MlEntriesResponse> MlClientImpl::getMlEntriesImpl(const TaskContextPtr& context,
        const std::string& ticket) const {
    using namespace http_getter;
    const auto request = get(config.location + "/apiv3/lists/info")
                        .headers(requestId=context->requestId(), serviceTicket=ticket)
                        .timeouts(config.httpOptions.timeouts.total, config.httpOptions.timeouts.connect)
                        .getArgs("fields"_arg="id,name,email")
                        .make();
    return perform(context, request).bind([&](const auto& response) {
        const auto httpStatusOk{200};
        if (response.status == httpStatusOk) {
            const std::string source{"ML"};
            return fromJson<MlEntriesResponse, Error>(response.body, context, source);
        } else {
            return make_expected_from_error<MlEntriesResponse>(error_code{services::Error::httpError});
        }
    });
}

expected<std::string> MlClientImpl::getTvmServiceTicket(const TaskContextPtr& context) const {
    std::string result;
    const auto ec{getTvm2Module()->get_service_ticket(config.targetServiceName, result)};
    if (ec) {
        LOGDOG_(context->logger(), error, log::message="failed to get TVM2 service ticket for ML request",
                log::error_code=ec);
        return make_unexpected(error_code{Error::tvmServiceTicketError, ec.message()});
    }

    return result;
}

expected<yhttp::response> MlClientImpl::perform(const TaskContextPtr& context,
        const http_getter::Request& request) const {
    auto trap{[](auto&& error) {
        if (error == services::Error::httpError) {
            return make_unexpected(error_code{services::Error::httpError,
                    "request to ML failed after all retries"});
        } else {
            return make_unexpected(std::move(error));
        }
    }};

    return performWithRetries(*getHttpClient(), context, request, makeRetryCondition(
            config.retries)).catch_error(std::move(trap));
}

} // namespace collie::services::ml
